import infra.callisto.controllers.shard.ctrl as ctrl
import infra.callisto.protos.deploy.tables_pb2 as tables_pb2  # noqa
import time


def test_stuck_states():
    states = [make_state(str(i)) for i in range(3)]
    update_target(states[::-1])

    push_status(states[0])

    update_target(states[::-1], progress_timeout=0)

    for state in states:
        assert actual(state)
        assert state.Target.PrepareTimestamp

    time.sleep(2)
    update_target(states[::-1], progress_timeout=0)

    for state in states[1:]:
        assert not actual(state)
    assert actual(states[0])

    states = states[:1]
    states.append(make_state("6"))
    states.append(make_state("7"))
    states.append(make_state("8"))

    update_target(states[::-1], progress_timeout=0)

    for state in states[1:]:
        assert actual(state)

    push_status(states[1])
    push_status(states[2])
    push_status(states[3])
    time.sleep(2)
    update_target(states[::-1], progress_timeout=0)
    for state in states:
        assert actual(state)

    push_status(states[1])
    assert states[1].Status.ActivateTimestamp
    assert not states[0].Status.ActivateTimestamp

    update_target(states[::-1], progress_timeout=0)
    assert not actual(states[0])
    assert actual(states[1])
    assert actual(states[2])


def test_dummy():
    states = [make_state('-1')]

    update_target(states)
    assert states[0].Target.PrepareTimestamp

    push_status(states[0])
    update_target(states)
    assert active(states[-1])

    push_status(states[0])
    update_target(states)
    assert actual(states[-1])

    states.insert(0, make_state('-2'))

    update_target(states)
    assert states[-2].Target.PrepareTimestamp
    assert actual(states[-1]) and stable_active(states[-1])

    push_status(states[0])
    update_target(states)
    assert actual(states[-2]) and active(states[-2])
    assert actual(states[-1]) and stable_active(states[-1])

    states.insert(0, make_state('-3'))
    update_target(states)
    assert states[-3].Target.PrepareTimestamp
    assert actual(states[-2]) and active(states[-2])
    assert actual(states[-1]) and stable_active(states[-1])

    push_status(states[0])
    update_target(states)
    assert actual(states[-3]) and active(states[-3])
    assert actual(states[-2]) and active(states[-2])
    assert actual(states[-1]) and stable_active(states[-1])

    assert not place_available(states)

    push_status(states[-2])
    update_target(states)
    assert actual(states[-3]) and active(states[-3])
    assert actual(states[-2]) and stable_active(states[-2])
    assert not actual(states[-1])

    assert place_available(states)


def test_drop_stuck_state():
    states = [make_state('-1')]

    update_target(states)
    assert actual(states[0])
    states[0].Target.PrepareTimestamp -= 20 * 60 + 1
    update_target(states)
    assert not actual(states[0])


def test_keep_last_stuck_state_with_no_presure():
    states = [make_state('-1')]

    update_target(states)
    assert actual(states[0])
    states[0].Target.PrepareTimestamp -= 20 * 60 + 1
    update_target(states, fresh_state_exists=False)
    assert actual(states[0])


def test_drop_skipped_state():
    states = [make_state('-1')]
    assert actual(states[0])

    states[0].Demand.Skip = True
    update_target(states)
    assert not actual(states[0])


def test_frozen_state():
    states = [
        make_state('-3'),
        make_state('-2'),
        make_state('-1'),
    ]
    states[-1].Demand.Active = True

    update_target(states)
    push_status(states[-3])
    push_status(states[-2])
    push_status(states[-1])

    update_target(states)
    push_status(states[-3])
    push_status(states[-2])
    push_status(states[-1])

    update_target(states)

    assert actual(states[-3]) and stable_active(states[-3])
    assert not actual(states[-2])
    assert actual(states[-1]) and stable_active(states[-1])


def test_fresh_state_provider():
    last_states = [
        make_state('-4', 'ns4'),
        make_state('-3', 'ns3'),
        make_state('-2', 'ns2'),
        make_state('-1', 'ns1'),
    ]

    fresh = get_fresh_state(last_states, 'ns2', last_known_state=None, freeze_enabled=False)
    control_state = last_states[-2]
    assert (fresh.Namespace, fresh.StateId) == (control_state.Namespace, control_state.StateId)

    fresh_again = get_fresh_state(last_states, 'ns2', last_known_state=fresh, freeze_enabled=False)
    assert fresh_again is None

    not_fresh = get_fresh_state(last_states, 'ns2', last_known_state=make_state('10', 'ns2'), freeze_enabled=False)
    assert not_fresh is None


def make_state(state_id, namespace='namespace'):
    return tables_pb2.TShardCtrlState(
        Namespace=namespace,
        StateId=state_id,
        Status=tables_pb2.TShardCtrlState.TStateEvents(CollectTimestamp=now()),
        Target=tables_pb2.TShardCtrlState.TStateEvents(CollectTimestamp=now())
    )


def push_status(state):
    if not state.Status.PrepareTimestamp and state.Target.PrepareTimestamp:
        state.Status.PrepareTimestamp = now()
    if not state.Status.ActivateTimestamp and state.Target.ActivateTimestamp:
        state.Status.ActivateTimestamp = now()


def now():
    return ctrl._now()  # noqa


def actual(state):
    return ctrl._is_actual(state)  # noqa


def active(state):
    return ctrl._is_active(state)  # noqa


def stable_active(state):
    return ctrl._is_stable_active(state)  # noqa


def place_available(states):
    return ctrl._is_place_available(states, ctrl.DEFAULT_STATES_LIMIT)  # noqa


def fresh(state):
    return ctrl._is_fresh(state)  # noqa


def update_target(states, fresh_state_exists=True, progress_timeout=ctrl.STARTUP_INTERVAL):
    ctrl._update_target(states, ctrl.DEFAULT_ACTIVE_STATES_LIMIT,
                        fresh_state_exists=fresh_state_exists, progress_timeout=progress_timeout)  # noqa


def get_fresh_state(last_states, namespace, last_known_state, freeze_enabled):
    return ctrl._get_fresh_state(last_states, namespace, last_known_state,
                                 freeze_enabled)  # noqa
