import infra.callisto.controllers.shard.ctrl as shardctrl
import infra.callisto.controllers.shard.tables as tables
import infra.callisto.controllers.utils.yp_utils as yp_utils
import infra.callisto.controllers.utils.gencfg_api as gencfg_api
import infra.callisto.libraries.yt as yt


def make_instance_providers(pod_sets, report_tags=None, cache_for=60):
    return [
        yp_utils.InstanceProvider(
            [yp_utils.DeploySlot(pod_set.Cluster, pod_set.PodSetId, port=80)],
            report_tags=report_tags,
            cache_for=cache_for)
        for pod_set in pod_sets
    ]


def make_gencfg_instance_providers(gencfg_group_set, mtn=True, report_tags=None):
    return [
        gencfg_api.InstanceProvider(
            [gencfg_api.GencfgGroup(group.Group, group.Tag, mtn=mtn)],
            report_tags=report_tags)
        for group in gencfg_group_set
    ]


def make_plutonium_fs(plutonium_fs, plutonium_fs_error_cluster_probability, use_rpc=True):
    return shardctrl.PlutoniumFS(
        yt.create_yt_client(plutonium_fs.MetaCluster, use_rpc=use_rpc),
        plutonium_fs.Path,
        plutonium_fs.ContentCluster,
        plutonium_fs.FallbackClusters or [],
        plutonium_fs_error_cluster_probability or 0
    )


def make_pod_target(deployers_target, readonly=True, use_rpc=True):
    return tables.PodTarget(
        yt.create_yt_client(deployers_target.Cluster, use_rpc=use_rpc),
        deployers_target.Path,
        readonly,
        deployers_target.TabletCellBundle or 'cajuper'
    )


def make_pod_status(deployers_status, use_rpc=True, readonly=True):
    return tables.PodStatus(
        yt.create_yt_client(deployers_status.Cluster, use_rpc=use_rpc),
        deployers_status.Path,
        readonly,
        deployers_status.TabletCellBundle or 'cajuper'
    )


def make_snapshot_replicator(config, readonly):
    yt_client = yt.create_yt_client(config.Cluster, use_rpc=True)
    return shardctrl.SnapshotReplicator(
        target_table=tables.SnapshotReplicatorTarget(
            yt_client,
            config.TargetTable,
            readonly,
            config.TabletCellBundle or 'cajuper',
        ),
        status_table=tables.SnapshotReplicatorStatus(
            yt_client,
            config.StatusTable,
            True,
            config.TabletCellBundle or 'cajuper',
        ),
    )


def make_coordinator(readonly, config):
    contour_states = tables.ShardCtrlState(
        yt.create_yt_client(config.CoordinatorStates.Cluster, use_rpc=True),
        config.CoordinatorStates.Path,
        readonly,
        config.CoordinatorStates.TabletCellBundle or 'cajuper'
    )

    snapshot_replicator = None
    if config.HasField('SnapshotReplicator'):
        snapshot_replicator = make_snapshot_replicator(config.SnapshotReplicator, readonly)

    banned_groups = tables.BannedGroupsStub()
    if config.HasField('BannedGroups'):
        banned_groups = tables.BannedGroups(
            yt.create_yt_client(config.BannedGroups.Cluster, use_rpc=True),
            config.BannedGroups.Path,
            readonly,
            config.BannedGroups.TabletCellBundle or 'cajuper'
        )

    return shardctrl.Coordinator(
        namespaces=config.Namespaces,
        public_states=tables.PublicStatesTable(
            yt.create_yt_client(config.PublicStates.Cluster, use_rpc=True),
            config.PublicStates.Path,
            readonly,
            config.PublicStates.TabletCellBundle or 'cajuper',
            ),
        locations=[
            shardctrl.Location(
                endpoint_set=location.HamsterEndpointSet if location.HasField('HamsterEndpointSet') else None,
                table=tables.ShardCtrlState(
                    yt.create_yt_client(location.State.Cluster, use_rpc=True),
                    location.State.Path,
                    readonly=True,
                    tablet_cell_bundle=location.State.TabletCellBundle or 'cajuper',
                )
            )
            for location in config.Locations
        ],
        contour_states=contour_states,
        snapshot_replicator=snapshot_replicator,
        banned_groups=banned_groups,
    )
