import collections
import datetime
import logging

import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.utils.funcs as funcs
import infra.callisto.controllers.sdk.request as request
import infra.callisto.controllers.sdk.blocks as blocks
import report
import state as slot_state


class _AbstractSearchController(sdk.Controller):
    _logger = logging.getLogger('search-ctrl')
    reports_alive_threshold = datetime.timedelta(minutes=5)

    def __init__(self, slot, agents, agent_shard_number_map, no_gencfg=False, namespace_prefix=None):
        super(_AbstractSearchController, self).__init__()
        self._slot = slot
        self._agents = agents
        self._agent_shard_number_map = agent_shard_number_map
        self._no_gencfg = no_gencfg
        self._target = slot_state.NullSlotState
        self._reports = {}
        self._namespace_prefix = namespace_prefix

    def update(self, reports):
        reports = _filter_reports(reports, self._agents)
        reports = funcs.imap_ignoring_exceptions(report.convert_report_to_searcher_report, reports)
        self._reports = {rep.agent: rep for rep in reports}

    def execute(self):
        if not self._target:
            self._logger.debug('%s: null target state', self)

    def gencfg(self):
        if self._no_gencfg:
            return {}
        configs = {}
        for agent, shard in self.target_agent_shard_mapping.items():
            configs[agent.host, agent.port] = {
                'alive': True,
                'shard': shard.fullname if shard else None,
            }
            if self._namespace_prefix and shard:
                configs[agent.host, agent.port]['resource'] = self._slot.shard_to_resource(
                    self._namespace_prefix,
                    shard
                ).to_dict()
        return configs

    @property
    def target_state(self):
        return self._target

    @target_state.setter
    def target_state(self, value):
        if self._target != value:
            self._logger.info('%s: %s -> %s', self, self._target, value)
            self._target = value

    @property
    def observed_agent_shard_mapping(self):
        mapping = {}
        for agent in self._agent_shard_number_map:
            if agent in self._reports:
                mapping[agent] = self._reports[agent].shard
            else:
                mapping[agent] = None
        return mapping

    @property
    def target_agent_shard_mapping(self):
        mapping = {}
        for agent, (group_number, shard_number) in self._agent_shard_number_map.items():
            if self._target:
                mapping[agent] = self._slot.make_shard(
                    group_number, shard_number,
                    self._target,
                )
            else:
                mapping[agent] = None
        return mapping

    def __str__(self):
        return '{}({})'.format(self.__class__.__name__, self._slot)


class SearchController(_AbstractSearchController):
    @property
    def tags(self):
        return {self._slot.group} if not self._no_gencfg else None  # FIXME

    def __init__(self, slot, agent_shard_number_map, no_gencfg=False, namespace_prefix=None):
        super(SearchController, self).__init__(
            slot,
            set(agent_shard_number_map.keys()),
            agent_shard_number_map,
            no_gencfg,
            namespace_prefix,
        )

    @property
    def observed_state(self):
        def _convert(mapping):
            return {agent: ({shard} if shard else {}) for agent, shard in mapping.items()}

        target = _convert(self.target_agent_shard_mapping)
        observed = _convert(self.observed_agent_shard_mapping)

        for state in _find_states(target, observed, self._slot):
            return state

        return slot_state.NullSlotState

    def json_view(self):
        return {
            'observed': self.observed_state.json(),
            'target': self.target_state.json(),
        }

    def diagnostics(self):
        def _convert(mapping):
            return {str(agent): ({str(shard)} if shard else {}) for agent, shard in mapping.items()}

        return {
            '1': _convert(self.observed_agent_shard_mapping),
        }

    def html_view(self):
        if not self.target_state:
            return blocks.SearchView(bars=[], name=self._slot.name)

        done, total = [], []
        targets, observed = self.target_agent_shard_mapping, self.observed_agent_shard_mapping
        for agent in targets:
            if not targets[agent]:
                continue
            total.append(targets[agent])
            if observed[agent] == targets[agent]:
                done.append(targets[agent])
        return blocks.SearchView(
            bars=[blocks.DoubleProgress(
                len(set(done)), len(set(total)), len(done), len(total),
                timestamp=self.target_state.timestamp
            )],
            name=self._slot.name,
        )


class _AbstractDeployerProxy(sdk.Controller):
    _logger = logging.getLogger('deployer-proxy')

    def __init__(self, slot, deploy_ctrl, agent_shard_number_map, namespace_prefix=None):
        super(_AbstractDeployerProxy, self).__init__()
        self._slot = slot
        self._tier = slot.tier
        self._deploy_ctrl = deploy_ctrl
        self._agent_shard_number_map = agent_shard_number_map
        self._targets = set()
        self._namespace_prefix = namespace_prefix

        deploy_ctrl.register(self)

    def agent_shards_target(self):
        agent_shards_map = collections.defaultdict(set)
        for agent, (group_number, shard_number) in self._agent_shard_number_map.items():
            for slot_state_ in self.target_states:
                shard = self._slot.make_shard(
                    group_number, shard_number,
                    slot_state_,
                )
                agent_shards_map[agent].add(shard)
        return agent_shards_map

    def host_shards_target(self):
        hosts_shards = collections.defaultdict(set)
        for agent, shards in self.agent_shards_target().items():
            hosts_shards[agent.node_name] |= shards
        return hosts_shards

    def agent_shards_observed(self):
        target = self.agent_shards_target()
        res = {
            agent: shards & self.observed_on_host(agent.node_name)
            for agent, shards in target.items()
        }
        return res

    @property
    def target_states(self):
        return self._targets.copy()

    @target_states.setter
    def target_states(self, targets):
        if targets != self._targets:
            self._logger.info('%s: %s -> %s', self, self._targets, targets)
            self._targets = targets

    def __str__(self):
        return '{}({})'.format(self.__class__.__name__, self._slot)

    # compatibility:
    def hosts_resources_target(self):
        assert self._namespace_prefix
        result = {}
        for host, shards in self.host_shards_target().iteritems():
            result[host] = self._slot.shards_to_resources(self._namespace_prefix, shards)
        return result

    def observed_on_host(self, host):
        if self._namespace_prefix:
            return self._slot.resources_to_shards(
                self._namespace_prefix,
                self._deploy_ctrl.observed_on_host(host, self._namespace_prefix)
            )
        else:
            return self._deploy_ctrl.observed_on_host(host)


class DeployerProxy(_AbstractDeployerProxy):
    @property
    def observed_states(self):
        return _find_states(
            self.agent_shards_target(),
            self.agent_shards_observed(),
            self._slot,
        )

    def json_view(self):
        return {
            'observed': [sl_st.json() for sl_st in self.observed_states],
            'target': [sl_st.json() for sl_st in self.target_states],
        }

    def html_view(self):
        timestamps = self.deploy_progress()

        if self._namespace_prefix:
            filter_arg = 'namespace=/{}/'.format(self._slot.tier.name)
        else:
            filter_arg = 'tier_name={}'.format(self._slot.tier.name)
        return blocks.DeployView(
            bars=[
                blocks.DoubleProgress(
                    len(set(state['done'])), len(set(state['total'])), len(state['done']), len(state['total']),
                    timestamp=timestamp
                )
                for timestamp, state in sorted(timestamps.items())
            ],
            name=self._slot.name,
            href_list=blocks.HrefList([
                blocks.Href(
                    'deploy-progress',
                    request.absolute_path(
                        request.ctrl_path(self._deploy_ctrl)
                    ) + '?handler=/deploy_progress&{}&viewer=1'.format(filter_arg)),
            ])
        )

    def deploy_progress(self):
        timestamps = collections.defaultdict(lambda: {'done': [], 'total': []})

        targets, observed = self.agent_shards_target(), self.agent_shards_observed()
        for agent in targets:
            for shard in targets[agent]:
                timestamps[shard.timestamp]['total'].append(shard)
                if shard in observed[agent]:
                    timestamps[shard.timestamp]['done'].append(shard)

        return timestamps


class SlotController(sdk.Controller):
    @property
    def path(self):
        return self.slot.name

    def __init__(self, slot, deployer, searcher):
        super(SlotController, self).__init__()
        self._slot = slot
        self._deployer = deployer
        self._searcher = searcher

        self.register(deployer, searcher)
        self.add_handler('/diagnostics', self._diagnostics)

    @property
    def slot(self):
        return self._slot

    def set_target_state(self, deployer_target_states, searcher_target_state):
        self._deployer.target_states = deployer_target_states
        self._searcher.target_state = searcher_target_state

    def get_observed_state(self):
        return self._deployer.observed_states, self._searcher.observed_state

    def get_target_state(self):
        return self._deployer.target_states, self._searcher.target_state

    def searchers_state(self):
        def _shard_to_json(shard):
            return shard.fullname if shard else None

        observed, target = self._searcher.observed_agent_shard_mapping, self._searcher.target_agent_shard_mapping
        return {
            agent.instance: {
                'target': _shard_to_json(target[agent]),
                'observed': _shard_to_json(observed[agent]),
            }
            for agent in target
        }

    def deploy_progress(self, timestamp=None):
        if timestamp:
            return {timestamp: self._deployer.deploy_progress().get(timestamp)}

        return self._deployer.deploy_progress()

    def json_view(self):
        return {
            'name': self.slot.name,
            'deployer': self._deployer.json_view(),
            'searcher': self._searcher.json_view(),
        }

    def html_view(self):
        return blocks.SlotView(self.slot.name, self._deployer.html_view(), self._searcher.html_view())

    def _diagnostics(self):
        return {'searcher': self._searcher.diagnostics(), 'deployer': self._deployer.diagnostics()}

    def __str__(self):
        return '{}({})'.format(self.__class__.__name__, self._slot)


def _filter_mapping(mapping, group_number):
    return {
        agent: number
        for agent, number in mapping.items()
        if number[0] == group_number
    }


def make_slot_controller(slot, deployer_ctrl, agent_shard_number_mapping, deploy_only=False, namespace_prefix=None):
    deployer = DeployerProxy(
        slot,
        deployer_ctrl,
        agent_shard_number_mapping,
        namespace_prefix=namespace_prefix,
    )
    searcher = SearchController(
        slot, agent_shard_number_mapping,
        no_gencfg=deploy_only,
        namespace_prefix=namespace_prefix,
    )
    return SlotController(slot, deployer, searcher)


def make_callisto_slot_controller(slot, deployer_ctrl, agent_shard_number_mapping, shard_group_number, namespace_prefix=None):
    agent_shard_number_mapping = _filter_mapping(agent_shard_number_mapping, shard_group_number)
    deployer = DeployerProxy(
        slot,
        deployer_ctrl,
        agent_shard_number_mapping,
        namespace_prefix=namespace_prefix,
    )
    searcher = SearchController(slot, agent_shard_number_mapping, namespace_prefix=namespace_prefix)
    return SlotController(slot, deployer, searcher)


def _map_shard_to_agents(agent_to_shards_map):
    result = collections.defaultdict(list)
    for agent, shards in agent_to_shards_map.items():
        for shard in shards:
            result[shard].append(agent)
    return result


def _find_states(target_agent_shard_map, observed_agent_shard_map, slot):
    target, observed = _map_shard_to_agents(target_agent_shard_map), _map_shard_to_agents(observed_agent_shard_map)

    if not target:
        return _find_states_fallback(observed_agent_shard_map, slot)

    observed_states, not_full_shards_cnt = set(), collections.defaultdict(int)
    for shard in filter(None, target):
        state = slot.slot_state_from_shard(shard)
        observed_states.add(state)
        if shard not in observed or not slot.is_shard_ready(len(observed[shard]), len(target[shard])):
            not_full_shards_cnt[state] += 1

    not_full_states = {
        state for state, count in not_full_shards_cnt.items()
        if not slot.is_ready(len(target) - count, len(target))
    }
    return observed_states - not_full_states


def _find_states_fallback(observed, slot):
    """might be helpful if callisto context was dropped"""
    total_agents_count = len(observed)
    observed = _map_shard_to_agents(observed)
    observed_states = collections.defaultdict(int)
    for shard in filter(None, observed):
        state = slot.slot_state_from_shard(shard)
        observed_states[state] += len(observed[shard])

    return {
        state for state in observed_states
        if slot.is_shard_ready(observed_states[state], total_agents_count)
    }


def _filter_reports(reports, known_agents):
    return (rep for agent, rep in reports.iteritems() if agent in known_agents)
