import logging

import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.search_source.controller as search_source
import infra.callisto.controllers.utils.bindings as bindings


class ExternalSlotsController(sdk.Controller):
    """
    This is a controller for automatic notifications only.
    No other controller machinery is used.
    """
    def __init__(self, url, readonly, target_table, status_table=None):
        super(ExternalSlotsController, self).__init__()
        self._url = url
        self._ctrl = bindings.SlotsController(url, readonly=readonly)
        self._readonly = readonly
        self._target_table = target_table
        self._status_table = status_table

    def notifications(self):
        head = self._target_table.head()
        if head and head.locked:
            return [
                sdk.notify.TextNotification(
                    message='ctrl table {} is locked.\n comment: {}'.format(self._target_table, head.comment),
                    level=sdk.notify.NotifyLevels.WARNING
                )
            ]
        return []

    def _basesearch_target(self, head, basesearch_timestamp):
        deploy_observed = self.get_deployer_observed_state()
        if basesearch_timestamp not in deploy_observed and head.target.search != basesearch_timestamp:
            return None
        return basesearch_timestamp

    def set_target_state(self, deployer_timestamps, basesearch_timestamp, null_if_not_deployed=False):
        head = self._target_table.head()
        if null_if_not_deployed:
            basesearch_timestamp = self._basesearch_target(head, basesearch_timestamp)
        new_target = search_source.Target(deployer_timestamps, basesearch_timestamp)
        if not head or head.target != new_target:
            if head and head.locked:
                logging.warning('table %s is locked! skip it', self._target_table)
            else:
                if self._readonly:
                    logging.debug('ro set %s: %s', self._url, new_target)
                else:
                    self._target_table.write(new_target, comment='external slots ctrl')
                    logging.debug('set %s: %s', self._url, new_target)
        else:
            logging.debug('%s: no diff; %s', self._url, new_target)

    def get_target_state(self):
        return self._ctrl.target_state()

    def get_observed_state(self):
        if self._status_table:
            head = self._status_table.head()
            logging.debug('[%s] Get observed state from yt: %s, %s', self._url, head.status.deploy, head.status.search)
            return head.status.deploy, head.status.search

        return self._ctrl.observed_state()

    def get_deployer_target_state(self):
        return self.get_target_state()[0]

    def get_deployer_observed_state(self):
        return self.get_observed_state()[0]

    def get_deploy_percentage(self):
        return self._ctrl.deploy_percentage()

    def json_view(self):
        try:
            deployer_ts, basesearch_ts = self._ctrl.observed_state()
            observed = {
                'deployer': list(deployer_ts),
                'searcher': basesearch_ts
            }
            state = ''
        except bindings.SlotsControllerException as ex:
            observed = None
            state = ex.message

        return {
            'url': self._url,
            'observed': observed,
            'state': state
        }
