from collections import namedtuple

import infra.callisto.controllers.utils.entities as entities
import infra.callisto.controllers.sdk.tier as tier


_Searcher = namedtuple('_Basesearch', ['host', 'port', 'shard', 'reported_at'])


class Searcher(_Searcher):
    def __new__(cls, host, port, shard, reported_at):
        assert shard is None or isinstance(shard, entities.Shard)
        return super(cls, cls).__new__(
            cls, str(host), int(port), shard, reported_at
        )

    @property
    def name(self):
        return '{}:{}'.format(self.host, self.port)

    @property
    def agent(self):
        return entities.Agent(self.host, self.port)

    def __str__(self):
        return 'Searcher(%s, %s)' % (self.host, self.port)

    def json(self):
        return dict(
            host=self.host,
            port=self.port,
            shard=self.shard.fullname if self.shard else None,
            reported_at=self.reported_at,
        )


def _find_shard(data):
    for shard_name, shard in data.iteritems():
        if shard.get('status') == 'RUN':
            return tier.parse_shard(shard_name)
    return None


def convert_report_to_searcher_report(report):
    return Searcher(
        report.host,
        report.port,
        _find_shard(report.data),
        report.generation_time,
    )
