import math
import logging

import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.utils.entities as entities

import state


class Slot(object):
    def __init__(
        self,
        name,
        tier,
        group,
        topology,
        use_mtn=False,
        ready_threshold=1,
    ):
        self.name = name
        self.tier = tier
        self.group = group
        self.topology = topology
        self.use_mtn = use_mtn
        self.ready_threshold = ready_threshold

    def make_shard(self, agent_group_number, agent_shard_number, slot_state):
        return self.tier.make_shard((agent_group_number, agent_shard_number), slot_state.timestamp)

    @classmethod
    def slot_state_from_shard(cls, shard):
        return state.SlotState(timestamp=shard.timestamp)

    def shard_to_resource(self, namespace_prefix, shard):
        assert shard.tier.name == self.tier.name
        return sdk.resource.shard_to_resource(namespace_prefix, shard)

    @classmethod
    def resource_to_shard(cls, namespace_prefix, resource):
        return sdk.resource.resource_to_shard(namespace_prefix, resource)

    def shards_to_resources(self, namespace_prefix, shards):
        return {
            self.shard_to_resource(namespace_prefix, shard)
            for shard in shards
        }

    @classmethod
    def resources_to_shards(cls, namespace_prefix, resources):
        return {
            cls.resource_to_shard(namespace_prefix, resource)
            for resource in resources
            if '/' not in resource.name
        }

    @classmethod
    def is_shard_ready(cls, observed_count, target_count):
        if target_count < 4:
            return observed_count >= 1
        if target_count == 4:
            return observed_count >= 2
        return observed_count >= math.floor(target_count * 0.75)

    def is_ready(self, observed_count, target_count):
        logging.debug('%s readiness: %s / %s  == %s, while required %s', self.name, observed_count, target_count, float(observed_count) / target_count, self.ready_threshold)
        return observed_count >= target_count * self.ready_threshold

    def __str__(self):
        return 'Slot(%s)' % self.name


class CallistoSlot(Slot):
    @classmethod
    def slot_state_from_shard(cls, shard):
        return state.SlotState(timestamp=shard.timestamp, group_number=shard.group_number)

    def make_shard(self, agent_group_number, agent_shard_number, slot_state):
        return self.tier.make_shard((slot_state.group_number, agent_shard_number), slot_state.timestamp)


class WebTier1Slot(Slot):
    @classmethod
    def shard_to_resource(cls, namespace_prefix, shard):
        return sdk.resource.chunk_to_resource(namespace_prefix, entities.Chunk(shard, 'local'))

    @classmethod
    def resource_to_shard(cls, namespace_prefix, resource):
        return sdk.resource.resource_to_chunk(namespace_prefix, resource).shard

    @classmethod
    def resources_to_shards(cls, namespace_prefix, resources):
        return {
            cls.resource_to_shard(namespace_prefix, resource)
            for resource in resources
            if resource.name.endswith('/local')
        }

    def is_ready(self, observed_count, target_count):
        return observed_count > target_count * 0.999


class WebTier1SlotFixedTimestamp(WebTier1Slot):
    def __init__(
            self,
            name,
            tier,
            group,
            topology,
            fixed_timestamp,
    ):
        super(WebTier1SlotFixedTimestamp, self).__init__(name, tier, group, topology)
        self._fixed_timestamp = fixed_timestamp

    def make_shard(self, agent_group_number, agent_shard_number, slot_state):
        return self.tier.make_shard((agent_group_number, agent_shard_number), self._fixed_timestamp)


class GeminiSlot(Slot):
    def is_ready(self, observed_count, target_count):
        return observed_count > target_count * 0.99


class RimSlot(Slot):
    def is_ready(self, observed_count, target_count):
        return observed_count > target_count * 0.97
