from collections import namedtuple


class _Keys(object):
    GroupNumber = 'group_number'
    Timestamp = 'timestamp'
    All = [Timestamp, GroupNumber]


_SlotState = namedtuple('SlotState', _Keys.All)


class SlotState(_SlotState):
    def __new__(cls, timestamp, group_number=None):
        if group_number is not None:
            group_number = int(group_number)
        if timestamp is not None:
            timestamp = int(timestamp)
        return super(cls, cls).__new__(cls, timestamp, group_number)

    def json(self):
        if not self:
            return None
        return {_Keys.GroupNumber: self.group_number, _Keys.Timestamp: self.timestamp}

    def __nonzero__(self):
        return self.timestamp is not None

    def __repr__(self):
        if not self:
            return 'NullSlotState'
        return 'SlotState({}: {})'.format(self.timestamp, self.group_number)


NullSlotState = SlotState(None, None)


def state_to_dict(state):
    return state.json()


def state_from_dict(state):
    if state is None:
        return NullSlotState
    return SlotState(
        timestamp=state[_Keys.Timestamp],
        group_number=state.get(_Keys.GroupNumber, None),
    )


def empty_mapping(contour):
    return {slot: NullSlotState for slot in contour.slots}


def mapping_to_dict(mapping):
    return {
        str(slot_number): state_to_dict(state)
        for slot_number, state in mapping.items()
    }


def mapping_from_dict(mapping):
    return {
        int(slot_number): state_from_dict(state)
        for slot_number, state in mapping.items()
    }
