import collections

import infra.callisto.controllers.build.extbuilder as extbuilder
import infra.callisto.controllers.build.innerbuilder as innerbuilder
import infra.callisto.controllers.deployer.controller as deploy_controller
import infra.callisto.controllers.deployer2.controller as deploy2_controller
import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.slots.controllers as controllers
import infra.callisto.controllers.utils.gencfg_api as gencfg_api

import intl2.controller as intl2_controller
import soft_controller


def make_controller(contour, yt_observer):
    deploy_ctrl = deploy_controller.make_deployer_controller2(
        [contour.deployer_group],
    )
    slot_controllers = _make_slots_ctrls(deploy_ctrl, contour.slots)
    intl2_ctrl = _make_intl2_ctrl(contour)

    if contour.builder_provider:
        builder = _make_builder_ctrl(contour.builder_provider, yt_observer)
    elif contour.builder_url:
        builder = extbuilder.ExternalBuilderObserver(contour.builder_url, yt_observer.tier)
    else:
        raise ValueError('provided neither builder-url nor groups')

    return SoftContourController(contour, deploy_ctrl, slot_controllers, intl2_ctrl, builder)


def make_controller2(contour, yt_observer, namespace_prefix, deploy_params=None):
    deploy_ctrl = deploy2_controller.make_controller2(
        [contour.deployer_group], deploy_params
    )

    slot_controllers = _make_slots_ctrls(deploy_ctrl, contour.slots, namespace_prefix)
    intl2_ctrl = _make_intl2_ctrl(contour)

    if contour.builder_provider:
        builder = _make_builder_ctrl(contour.builder_provider, yt_observer)
    elif contour.builder_url:
        builder = extbuilder.ExternalBuilderObserver(contour.builder_url, yt_observer.tier)
    else:
        raise ValueError('provided neither builder-url nor groups')

    return SoftContourController(contour, deploy_ctrl, slot_controllers, intl2_ctrl, builder)


def _make_slots_ctrls(deploy_ctrl, slots, namespace_prefix=None):
    return {
        shard_group_number: controllers.make_callisto_slot_controller(
            slot,
            deploy_ctrl,
            gencfg_api.agent_shard_number_mapping(slot.group, slot.topology, mtn=slot.use_mtn),
            shard_group_number,
            namespace_prefix=namespace_prefix,
        ) for shard_group_number, slot in slots.items()
    }


def _make_intl2_ctrl(contour):
    intl2_groups = [contour.intl2_group]
    if contour.hamster:
        intl2_groups.append(contour.hamster.intl2_group)

    intl2_agents = gencfg_api.get_agents(intl2_groups)

    return intl2_controller.IntL2Controller(
        contour,
        intl2_agents,
        max_switching=3,
        unsafe=False,
    )


def _make_builder_ctrl(builder_provider, yt_observer):
    return innerbuilder.make_controller(
        yt_observer,
        builder_provider,
        assist_last_n_on_finish=5,
        space_needed_full_build_coeff=1,
        space_needed_inc_build_coeff=1,
    )


class SoftContourController(sdk.Controller):
    @property
    def path(self):
        return self._contour.name

    def __init__(self, contour, deploy_ctrl, slot_controllers, intl2_ctrl, builder):
        super(SoftContourController, self).__init__()
        self._contour = contour
        self._deploy_ctrl = deploy_ctrl
        self._builder = builder
        self._intl2_ctrl = intl2_ctrl
        self._soft_ctrl = soft_controller.SoftController(contour, slot_controllers)
        self.add_handler('/shards_on_search', self._shards_on_search)

        self.register(
            self._deploy_ctrl,
            self._builder,
            self._intl2_ctrl,
            self._soft_ctrl,
        )

    def execute(self):
        self._soft_ctrl.update_targets(self._builder.prepared(), self._intl2_ctrl.unused_slots())
        self._intl2_ctrl.update_state(self._soft_ctrl.slots_state())

    def deploy_progress(self, tier_name=None):
        return self._deploy_ctrl.deploy_progress(tier_name)

    def build_progress(self, tier_name):
        return self._builder.build_progress(tier_name)

    def searchers_state(self, slot_name=None):
        return self._soft_ctrl.searchers_state(slot_name)

    def json_view(self):
        view = collections.OrderedDict()
        view['contour'] = self._contour.name
        view['tier'] = self._contour.tier.name
        view['intl2'] = self._intl2_ctrl.json_view()
        view['build'] = self._builder.json_view()
        view['slots'] = self._soft_ctrl.json_view()
        return view

    def _shards_on_search(self):
        """used by jupiter"""
        result = []
        for slot_name, slot_state in self._soft_ctrl.slots_state().items():
            result.append({
                'group_number': slot_state.group_number,
                'timestamp': slot_state.timestamp,
                'observed_intl2_ratio': self._intl2_ctrl.observed_slot_usage(slot_name),
            })
        return result
