import collections
import infra.callisto.controllers.shard.ctrl as shardctrl
from infra.callisto.controllers.shard import entities
from infra.callisto.controllers.shard.match_banned import match_banned_group
import infra.callisto.controllers.shard.tables as tables
import infra.callisto.controllers.sdk.registry as registry
import infra.callisto.libraries.yt as yt
import infra.callisto.protos.deploy.tables_pb2 as tables_pb2
import search.plutonium.deploy.proto.sources_pb2 as sources_pb2

from infra.callisto.controllers.shard.utils import make_coordinator, make_instance_providers, make_pod_target, make_pod_status


Topology = collections.namedtuple('Topology', ['Namespace', 'StateId', 'ResourceSpec'])


class ChunkCtrlMock(object):
    def __init__(self, plutonium_fs):
        self.plutonium_fs = plutonium_fs

    def is_active(self, state):
        return True

    def set_target(self, states):
        pass

    def update_status(self, namespaces):
        pass

    def get_topology(self, states):
        shardctrl._log.info(states)

        return [
            Topology(
                state.Namespace,
                state.StateId,
                sources_pb2.TSource(
                    Compound=self.plutonium_fs.get_mappings_resource(state.Namespace, state.StateId)
                )
            )
            for state in states
        ]


class AdvMachineDaemonTarget(entities._AbstractTarget):
    namespace = 'adv_machine_banner'


class AdvMachineDaemon(object):
    def __init__(self, pods_providers, target_table, status_table,
                 max_not_ready_pods=2,
                 max_bad_pods=10,
                 min_mark_bad_seconds=10,
                 max_mark_bad_seconds=60,
                 min_ready_threshold=0.75):
        self._pods_providers = pods_providers
        self._target_table = target_table
        self._status_table = status_table
        self._bad_pods = shardctrl.BadPods(max_bad_pods, min_mark_bad_seconds, max_mark_bad_seconds)
        self._min_ready_threshold = min_ready_threshold
        self._ignored_pods = []
        self._dead_pods = []
        self._max_not_ready_pods = max_not_ready_pods
        self._switching_percent = 1.0

        self._status = collections.defaultdict(
            lambda: {'active_pods': set(), 'prepared_pods': self._list_pods_ids()}
        )

    def update_status(self, banned_sets=()):
        ready_status = (tables_pb2.EDownloadState.ACTIVE,)
        target_keys = [
            {'PodId': status.PodId, 'Namespace': status.Namespace, 'LocalPath': status.LocalPath}
            for status in self._status_table.list([AdvMachineDaemonTarget.namespace], self._list_pods_ids(banned_sets))
            if status.ResourceState.Status in ready_status
        ]
        shardctrl.global_timer.stamp_delta('AdvMachine pods statuses ready')

        result = collections.defaultdict(self._status.default_factory)
        for target in self._target_table.lookup(target_keys):
            if target.Namespace == AdvMachineDaemonTarget.namespace:
                key = AdvMachineDaemonTarget.get_snapshot(target)
                result[key]['active_pods'].add(target.PodId)
        shardctrl.global_timer.stamp_delta('AdvMachine status ready')

        self._status = result

    def is_prepared(self, state):
        return True

    def prepare(self, states):
        pass

    def is_active(self, state, realtime_config, banned_sets=()):
        check_pods = self._list_pods_ids(banned_sets)
        alive_pods = self._list_alive_pods(banned_sets)
        ready_pods = self._status[state.Namespace, state.StateId]['active_pods']
        waiting_pods = list()
        self._ignored_pods = list(check_pods - alive_pods - ready_pods)
        self._dead_pods = list(check_pods - alive_pods)
        for pod in list(ready_pods):
            self._bad_pods.mark_ok(pod)
        for pod in list(alive_pods - ready_pods):
            if not self._bad_pods.try_mark_bad(pod):
                waiting_pods.append(pod)
        ready_percent = len(ready_pods) / float(len(check_pods))
        if len(waiting_pods) > realtime_config.get("saas2_adv_machine_max_not_ready_pods", self._max_not_ready_pods) \
                or ready_percent < realtime_config.get("saas2_adv_machine_min_ready_threshold", self._min_ready_threshold):
            sample_size = 3
            shardctrl._log.debug(
                'State (%s, %s) is still not active. Waiting for %s [ready on %s / %s]',
                state.Namespace, state.StateId, waiting_pods[:sample_size],
                len(ready_pods), len(check_pods)
            )
            return False
        else:
            self._switching_percent = ready_percent
            return True

    def get_bad_pods(self):
        return self._bad_pods.get_bad_pods()

    def get_ignored_pods(self):
        return self._ignored_pods

    def get_dead_pods(self):
        return self._dead_pods

    def get_switching_percent(self):
        return self._switching_percent

    def activate(self, topologies):
        resources = []
        pods = self._list_pods_ids()
        for topology in topologies:
            target = AdvMachineDaemonTarget(topology.Namespace, topology.StateId, resource_spec=topology.ResourceSpec)
            resources.extend(target.get_targets(pods))

        if resources:
            shardctrl._log.debug('%s resources for deliver', len(resources))
            self.deliver(resources)

    def deliver(self, resources):
        self._target_table.update(resources)

    def _list_providers(self, banned_sets):
        result = []
        for provider in self._pods_providers:
            if match_banned_group(provider, banned_sets):
                shardctrl._log.debug('skip provider because %s banned', str(provider.group_keys()))
            else:
                result.append(provider)
        return result

    def _list_alive_pods(self, banned_sets=()):
        pods = set()
        for provider in self._list_providers(banned_sets):
            for agent in provider.agents_instances.itervalues():
                if agent.is_alive:
                    pods.add(agent.id)
        return pods

    def _list_pods_ids(self, banned_sets=()):
        return {
            pod_id
            for provider in self._list_providers(banned_sets)
            for pod_id in provider.ids
        }


def make_controller(readonly, config):
    local_states = tables.ShardCtrlState(
        yt.create_yt_client(config.ShardCtrlState.Cluster, use_rpc=True),
        config.ShardCtrlState.Path,
        readonly,
        config.ShardCtrlState.TabletCellBundle or 'cajuper'
    )

    chunk_ctrl = ChunkCtrlMock(
        shardctrl.PlutoniumFS(
            yt.create_yt_client(config.PlutoniumFS.MetaCluster, use_rpc=True),
            config.PlutoniumFS.Path,
            config.PlutoniumFS.ContentCluster,
            config.PlutoniumFS.FallbackClusters or [],
            config.PlutoniumFSErrorClusterProbability or 0,
            'TestTier/'
        )
    )

    banned_groups = tables.BannedGroupsStub()

    orchestrated_service = AdvMachineDaemon(
        make_instance_providers(config.BkStat.PodSet),  # TODO: AdvMachineStat?
        make_pod_target(config.DeployersTarget, readonly),
        make_pod_status(config.DeployersStatus, readonly=readonly),
    )

    return shardctrl.ShardCtrl(
        namespaces=config.Namespaces,
        chunk_ctrl=chunk_ctrl,
        orchestrated_service=orchestrated_service,
        states=local_states,
        contour_states=tables.ShardCtrlState(
            yt.create_yt_client(config.CoordinatorStates.Cluster, use_rpc=True),
            config.CoordinatorStates.Path,
            readonly,
            config.CoordinatorStates.TabletCellBundle or 'cajuper'
        ),
        realtime_config_table=tables.RealtimeConfig(
            yt.create_yt_client(config.RealtimeConfig.Cluster, use_rpc=True),
            config.RealtimeConfig.Path,
            readonly,
            config.RealtimeConfig.TabletCellBundle or 'cajuper'
        ),
        enable_freeze=config.EnableFreezing or False,
        states_limit=config.StatesLimit or shardctrl.DEFAULT_STATES_LIMIT,
        progress_timeout=config.StateProgressTimeout or shardctrl.STARTUP_INTERVAL,
        banned_groups=banned_groups
    )


registry.register2('adv_machine/saas2/shard', make_controller, sleep_time=4, config_type=shardctrl.TShardConfig)
registry.register2('adv_machine/saas2/coordinator', make_coordinator, sleep_time=4, config_type=shardctrl.TCoordinatorConfig)
