import collections
import logging

import infra.callisto.controllers.build.innerbuilder as builder
import infra.callisto.controllers.sdk.registry as registry
import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.search_source.controller as search_source
import infra.callisto.controllers.slots.extctrl as extctrl
import infra.callisto.controllers.utils.yp_utils as yp_utils
import infra.callisto.libraries.yt as yt_utils

import location_observer_ctrl
import yp_location_ctrl
import yt_observers
import commercial_consts as consts


def get_locational_ctrl_url(location):
    return "{}/{}".format(consts.CLUSTERSTATE_COMMERCIAL_URL, location)


def get_yt_status_table_prod(location, readonly):
    yt_client = yt_utils.create_yt_client(consts.YT_PROXY, use_rpc=True)
    return search_source.get_yt_status_table_absolute_path(yt_client, '{}/{}/status'.format(consts.YT_PROD_ROOT, location), readonly)


def get_yt_target_table_prod(location, readonly):
    yt_client = yt_utils.create_yt_client(consts.YT_PROXY, use_rpc=True)
    return search_source.get_yt_target_table_absolute_path(yt_client, '{}/{}/target'.format(consts.YT_PROD_ROOT, location), readonly)


def get_yt_status_table_pip(readonly):
    yt_client = yt_utils.create_yt_client(consts.YT_PROXY, use_rpc=True)
    return search_source.get_yt_status_table_absolute_path(yt_client, '{}/status'.format(consts.YT_PIP_ROOT), readonly)


def get_yt_target_table_pip(readonly):
    yt_client = yt_utils.create_yt_client(consts.YT_PROXY, use_rpc=True)
    return search_source.get_yt_target_table_absolute_path(yt_client, '{}/target'.format(consts.YT_PIP_ROOT), readonly)


class MainCtrl(sdk.Controller):
    path = consts.MAIN_CTRL_NAME

    def __str__(self):
        return "Main images commercial controller ({0})".format(self.path)

    def __init__(self, readonly):
        super(MainCtrl, self).__init__()
        self.readonly = readonly

        self._prod_ruler = make_ruler()
        self.state_observer = location_observer_ctrl.CommercialLocationObserverCtrl()
        self.builder = builder.make_controller(
            yt_observers.ImgCommercialTier0,
            instance_provider=yp_utils.InstanceProvider(
                [yp_utils.DeploySlot(consts.BUILDER_LOCATION, consts.BUILDER_ENDPOINTSET, consts.BUILDER_PORT)],
                report_tags=consts.BUILDER_REPORT_TAGS,
                cache_for=1800
            ),
            space_needed_full_build_coeff=1.0,
            space_needed_inc_build_coeff=0.3,
            assist_last_n_on_finish=30,
        )

        self.pip_slots = extctrl.ExternalSlotsController(get_locational_ctrl_url('pip'), readonly, get_yt_target_table_pip(readonly))
        # self.man_slots = extctrl.ExternalSlotsController(get_locational_ctrl_url('man'), readonly, get_yt_target_table_prod('man', readonly))
        self.sas_slots = extctrl.ExternalSlotsController(get_locational_ctrl_url('sas'), readonly, get_yt_target_table_prod('sas', readonly))
        self.vla_slots = extctrl.ExternalSlotsController(get_locational_ctrl_url('vla'), readonly, get_yt_target_table_prod('vla', readonly))

        self.register(
            self.state_observer,
            self.builder,
            self.pip_slots,
            # self.man_slots,
            self.sas_slots,
            self.vla_slots,
        )

    def execute(self):
        if self.state_observer.common_state:
            prod_common_timestamp = int(self.state_observer.common_state)

            logging.debug('prod_common_timestamp: {prod_common_timestamp}'.format(prod_common_timestamp=prod_common_timestamp))
            self._set_pip(prod_common_timestamp)
            self._set_prod(prod_common_timestamp)
        else:
            logging.debug('No common state observed, leaving pip & prod intact')

    def _set_pip(self, prod_common_timestamp):
        # up to 2 unskipped timestamps from build_states after prod_common_timestamp, or itself
        deploy_target = self._prod_ruler.to_deploy_on_acceptance(prod_common_timestamp)
        # min timestamp > prod_common_timestamp from deploy_target, or prod_common_timestamp
        basesearch_target = self._prod_ruler.search_target_on_acceptance(deploy_target, prod_common_timestamp)
        # write new timestamp to the target table
        logging.debug('pip: set {deploy_target}, {basesearch_target}'.format(deploy_target=deploy_target, basesearch_target=basesearch_target))
        self.pip_slots.set_target_state(deploy_target, basesearch_target, null_if_not_deployed=True)

    def _set_prod(self, prod_common_timestamp):
        # 2 unskipped timestamps from build_states starting from prod_common_timestamp, or the last 2 timestamps if the last one is prod
        deploy_target = self._prod_ruler.to_deploy_on_prod(prod_common_timestamp)
        assert prod_common_timestamp in deploy_target

        # set target state in prod target tables of locational controllers
        for index, slots_controller in enumerate((self.sas_slots, self.vla_slots)):
            logging.debug('prod {index}: set {deploy_target}'.format(index=index, deploy_target=deploy_target))
            slots_controller.set_target_state(deploy_target, prod_common_timestamp)

    def json_view(self):
        result = collections.OrderedDict()
        result['__readonly__'] = self.readonly
        # result['man_commercial'] = self.man_slots.json_view()
        result['sas_commercial'] = self.sas_slots.json_view()
        result['vla_commercial'] = self.vla_slots.json_view()
        result['pip_commercial'] = self.pip_slots.json_view()
        result['builders'] = {
            self.builder.tier.name: self.builder.json_view()
        }
        result['prod'] = self.state_observer.json_view()

        return result

    def build_progress(self, tier_name):
        return self.builder.build_progress(tier_name)


def make_ruler():
    acceptance_source = sdk.prod_ruler.AcceptanceTableSource(
        path=consts.YT_ACCEPTANCE_RESULT_TABLE,
        yt_client=yt_utils.create_yt_client(consts.YT_PROXY)
    )
    return sdk.prod_ruler.DeployRuler(
        build_source=yt_observers.ImgCommercialTier0.source,
        acceptance_source=acceptance_source,
        max_states_to_keep=2,
    )


def make_main_controller(readonly):
    return MainCtrl(readonly)


def make_yp_location_ctrl_pip_params(readonly):
    return yp_location_ctrl.MakeYpControllerParams(
        name=consts.LOCATION_CTRL_NAME_TEMPLATE.format('pip'),
        location=consts.PIP_LOCATION,
        deploy_unit_id=consts.DEPLOY_UNIT_ID_PIP,
        slot_name=consts.SLOT_NAME_PIP,
        tier=consts.TIER,
        namespace=consts.NAMESPACE,
        target_table=get_yt_target_table_pip(readonly),
        status_table=get_yt_status_table_pip(readonly),
        is_pip=True,
    )


def make_yp_location_ctrl_prod_params(location, readonly):
    return yp_location_ctrl.MakeYpControllerParams(
        name=consts.LOCATION_CTRL_NAME_TEMPLATE.format(location),
        location=location,
        deploy_unit_id=consts.DEPLOY_UNIT_IDS_PROD[location],
        slot_name=consts.SLOT_NAME_PROD,
        tier=consts.TIER,
        namespace=consts.NAMESPACE,
        target_table=get_yt_target_table_prod(location, readonly),
        status_table=get_yt_status_table_prod(location, readonly),
        is_pip=False,
    )


def make_pip_controller(readonly):
    return yp_location_ctrl.make_yp_controller(make_yp_location_ctrl_pip_params(readonly))


def make_sas_controller(readonly):
    return yp_location_ctrl.make_yp_controller(make_yp_location_ctrl_prod_params('sas', readonly))


def make_man_controller(readonly):
    return yp_location_ctrl.make_yp_controller(make_yp_location_ctrl_prod_params('man', readonly))


def make_vla_controller(readonly):
    return yp_location_ctrl.make_yp_controller(make_yp_location_ctrl_prod_params('vla', readonly))


registry.register('images/prod/commercial', make_main_controller, [registry.ReportsBackends.V2.vla, registry.ReportsBackends.V2.sas])
registry.register('images/prod/commercial/acceptance', make_pip_controller, [registry.ReportsBackends.V2.sas])
registry.register('images/prod/commercial/sas', make_sas_controller, [registry.ReportsBackends.V2.sas])
# registry.register('images/prod/commercial/man', make_man_controller, [registry.ReportsBackends.V2.man])
registry.register('images/prod/commercial/vla', make_vla_controller, [registry.ReportsBackends.V2.vla])
