import collections
import datetime
import logging

import infra.callisto.controllers.build.innerbuilder as innerbuilder
import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.sdk.notify as notify
import infra.callisto.controllers.sdk.registry as registry
import infra.callisto.controllers.sdk.tier as tier
import infra.callisto.controllers.slots.extctrl as extctrl
import infra.callisto.libraries.yt as yt_utils
import infra.callisto.controllers.utils.gencfg_api as gencfg_api

import oldprod2
import pip
import sas
# import man
import vla
import yt_observers
import yt_copier


class Ctrl(sdk.Controller):
    path = 'images'

    def __init__(self, readonly):
        super(Ctrl, self).__init__()

        self.readonly = readonly

        builders_default_topology = 'stable-154-r3599'
        self.builders = {
            tier.ImgTier0: innerbuilder.make_controller(
                yt_observers.ImgTier0,
                instance_provider=gencfg_api.InstanceProvider(
                    [gencfg_api.GencfgGroup('VLA_IMGS_TIER0_BUILD', builders_default_topology, mtn=True)],
                    report_tags={'VLA_IMGS_TIER0_BUILD'}
                ),
                space_needed_full_build_coeff=1.0,
                space_needed_inc_build_coeff=0.3,
                assist_last_n_on_finish=15,
                remove_failed_tasks=False,
            ),
            tier.ImgTier1: innerbuilder.make_controller(
                yt_observers.ImgTier1,
                instance_provider=gencfg_api.InstanceProvider(
                    [gencfg_api.GencfgGroup('VLA_IMGS_TIER1_BUILD', builders_default_topology, mtn=True)],
                    report_tags={'VLA_IMGS_TIER1_BUILD'}
                ),
                space_needed_full_build_coeff=1.0,
                space_needed_inc_build_coeff=0.3,
                assist_last_n_on_finish=5,
                remove_failed_tasks=False,
            ),
            tier.ImgEmbeddingTier0: innerbuilder.make_controller(
                yt_observers.ImgEmbeddingTier0,
                instance_provider=gencfg_api.InstanceProvider(
                    [gencfg_api.GencfgGroup('VLA_IMGS_EMBEDDING_BUILD', builders_default_topology, mtn=True)],
                    report_tags={'VLA_IMGS_EMBEDDING_BUILD'}
                ),
                space_needed_full_build_coeff=1.0,
                space_needed_inc_build_coeff=0.3,
                assist_last_n_on_finish=5,
                remove_failed_tasks=False,
            ),
            tier.ImgInvIndexTier0: innerbuilder.make_controller(
                yt_observers.ImgInvIndexTier0,
                instance_provider=gencfg_api.InstanceProvider(
                    [gencfg_api.GencfgGroup('VLA_IMGS_INVERTED_INDEX_BUILD', 'stable-157-r772', mtn=True)],
                    report_tags={'VLA_IMGS_INVERTED_INDEX_BUILD'}
                ),
                space_needed_full_build_coeff=1.0,
                space_needed_inc_build_coeff=0.3,
                assist_last_n_on_finish=5,
                remove_failed_tasks=False,
            ),
            tier.ImgMmetaTier0: innerbuilder.make_controller(
                yt_observers.ImgMmetaTier0,
                instance_provider=gencfg_api.InstanceProvider(
                    [gencfg_api.GencfgGroup('VLA_IMGS_MMETA_BUILD', builders_default_topology, mtn=True)],
                    report_tags={'VLA_IMGS_MMETA_BUILD'}
                ),
                assist_last_n_on_finish=1,
                remove_failed_tasks=False,
            ),
        }

        self._prod_state_table = oldprod2.get_status_table(readonly=True)

        self.pip = extctrl.ExternalSlotsController(_ctrls + '/img/pip', readonly, pip.get_yt_target_table(readonly),
                                                   pip.get_yt_status_table(readonly=True))
        self.sas = extctrl.ExternalSlotsController(_ctrls + '/img/prod-sas', readonly, sas.get_yt_target_table(readonly),
                                                   sas.get_yt_status_table(readonly=True))
        # self.man = extctrl.ExternalSlotsController(_ctrls + '/img/prod-man', readonly, man.get_yt_target_table(readonly),
        #                                            man.get_yt_status_table(readonly=True))
        self.vla = extctrl.ExternalSlotsController(_ctrls + '/img/prod-vla', readonly, vla.get_yt_target_table(readonly),
                                                   vla.get_yt_status_table(readonly=True))

        yt_client = yt_utils.create_yt_client('arnold')
        self.yt_copiers = {
            tier.ImgTier0: yt_copier.ImagesChunksCopierController(
                yt_observers.ImgTier0,
                'images/prod/yt',
                yt_client,
                readonly
            ),
            "chunk-ctl-supplier": yt_copier.ImagesChunkCtlSpecsController(
                yt_observers.ImgTier0,
                yt_client,
                readonly
            ),
        }

        self.register(
            self.pip,
            self.sas,
            # self.man,
            self.vla,
            *(self.yt_copiers.values() + self.builders.values())
        )
        self.deploy_ruler = make_deploy_ruler()
        self.switcher = make_switcher(self.readonly)
        self._notifications = notify.NotificationsAggregator()

    def execute(self):
        if self._prod_state_table.common_state:
            prod_timestamp = int(self._prod_state_table.common_state)

            if 17 <= datetime.datetime.now().hour <= 23:
                self._set_prod(prod_timestamp)

            self._set_pip(prod_timestamp)
            self._notifications.add_notifications(self.switcher.check_and_switch(prod_timestamp))
        else:
            logging.debug('No common state observed, leaving pip & prod intact')

        self._notifications.push_buffer()

    def _common_builders_timestamps(self):
        return set.intersection(*[builder.timestamps() for builder in self.builders.values()])

    def _set_pip(self, prod_timestamp):
        deploy_target = self.deploy_ruler.to_deploy_on_acceptance(prod_timestamp)
        basesearch_target = self.deploy_ruler.search_target_on_acceptance(deploy_target, prod_timestamp)
        self.pip.set_target_state(deploy_target, basesearch_target, null_if_not_deployed=True)

    def _set_prod(self, prod_timestamp):
        deploy_target = self.deploy_ruler.to_deploy_on_prod(prod_timestamp)
        assert prod_timestamp in deploy_target

        for index, location in enumerate((self.sas, self.vla)):
            if index * 15 < datetime.datetime.now().minute < (index + 1) * 15:
                location.set_target_state(deploy_target, prod_timestamp)

    def json_view(self):
        result = collections.OrderedDict()
        result['__readonly__'] = self.readonly
        # result['man'] = self.man.json_view()
        result['sas'] = self.sas.json_view()
        result['vla'] = self.vla.json_view()
        result['pip'] = self.pip.json_view()
        result['prod'] = {}
        result['builders'] = {
            tier_.name: builder.json_view() for tier_, builder in self.builders.items()
        }

        return result

    def build_progress(self, tier_name):
        for _tier in self.builders:
            if _tier.name == tier_name:
                return self.builders[_tier].build_progress(tier_name)

    def notifications(self):
        return self._notifications.get_notifications()


def get_acceptance_table():
    return sdk.prod_ruler.AcceptanceTableSource(
        path='//home/images/shard_deploy/acceptance_result',
        yt_client=yt_utils.create_yt_client('arnold')
    )


def make_deploy_ruler():
    return sdk.prod_ruler.DeployRuler(
        build_source=yt_observers.Sources.ImagesTier0Source,
        acceptance_source=get_acceptance_table(),
        max_states_to_keep=2,
    )


def make_search_ruler():
    return sdk.prod_ruler.SearchRuler(
        acceptance_source=get_acceptance_table(),
        target_tables=[
            location.get_yt_target_table(readonly=True)
            for location in (sas, vla)
        ],
        status_tables=[
            location.get_yt_status_table(readonly=True)
            for location in (sas, vla)
        ],
    )


def make_switcher(readonly):
    return sdk.prod_ruler.Switcher(
        search_ruler=make_search_ruler(),
        task_type='SWITCH_IMAGES_DB_INDEX',
        vertical_name='images',
        should_be_deployed_time=datetime.time(hour=22),
        enable_autorelease=True,
        readonly=readonly,
    )


def make_img_controller(readonly):
    return Ctrl(readonly)


_ctrls = 'http://ctrl.clusterstate.yandex-team.ru'


registry.register('images/prod/main', make_img_controller, [registry.ReportsBackends.V2.vla])
