import collections
import logging

import infra.callisto.controllers.utils.sandbox_utils as sandbox_utils
import infra.callisto.controllers.utils.secrets as secrets
import infra.nanny.nanny_services_rest.nanny_services_rest.client as nanny


class LocationObserver(object):
    def __init__(self, base_service, files_to_check):
        self.base_service = base_service
        self.files_to_check = files_to_check

        self.state = None
        self._state_resource = None
        self._yt_state = None

    def update(self):
        self._update_yt_state()

        self.state = LocationState(
            yt_state=self._yt_state,
            config=LocationConfig(
                base=_get_service_sandbox_resources_by_local_paths(self.base_service, self.files_to_check),
            )
        )

    def _update_yt_state(self):
        try:
            self._state_resource, self._yt_state = self._get_yt_state(self.base_service)
        except:
            logging.debug('Could not update yt state for %s', self.base_service)

    def _get_yt_state(self, base_service):
        try:
            current_state_resource = _get_vars_conf(base_service)
        except Exception as e:
            logging.debug('Could not get vars.conf resource from service %s, error: %s', base_service, str(e))
            raise e

        if not current_state_resource:
            logging.debug('Could not get current_state_resource from service %s', base_service)

        if current_state_resource and self._state_resource != current_state_resource:
            logging.info('Found new state resource (vars.conf) for service %s: sbr:%s', base_service, current_state_resource['resource_id'])

            try:
                task_details = sandbox_utils.get_task(current_state_resource['task_id'])
            except Exception as e:
                logging.debug('Could not get sandbox info of vars.conf task %s, error: %s', current_state_resource['task_id'], str(e))
                raise e

            try:
                current_db_timestamp = task_details.get('input_parameters', {}).get('db_timestamp')
            except Exception as e:
                logging.debug('Could not get current_db_timestamp from parameters of vars.conf task %s, error: %s', current_state_resource['task_id'], str(e))
                raise e

            if not current_db_timestamp:
                logging.debug('Could not get current_db_timestamp for %s', base_service)
                raise RuntimeError('Could not get yt state for %s', base_service)

            logging.debug('Got current_db_timestamp from parameters of vars.conf task %s: %s', current_state_resource['task_id'], current_db_timestamp)
            return current_state_resource, str(current_db_timestamp)
        else:
            if not current_state_resource:
                logging.debug('current_state_resource is empty, default to previous values: %s, %s', str(self._state_resource), str(self._yt_state))
            return self._state_resource, self._yt_state


LocationState = collections.namedtuple('LocationState', ['yt_state', 'config'])
LocationConfig = collections.namedtuple('LocationConfig', ['base'])


def _get_service_sandbox_resources_by_local_paths(service, local_paths):
    def filter_fields(nanny_dict):
        return {key: nanny_dict.get(key) for key in {'resource_id', 'task_id'}}

    result = dict.fromkeys(local_paths)

    for sandbox_file in _get_service_sandbox_files(service):
        if sandbox_file['local_path'] in result:
            result[sandbox_file['local_path']] = filter_fields(sandbox_file)

    return result


def _get_vars_conf(service):
    for sandbox_file in _get_service_sandbox_files(service):
        if sandbox_file['local_path'] == 'vars.conf':
            return sandbox_file


def _get_service_sandbox_files(service):
    return _get_service_resources(service)['sandbox_files']


def _get_service_resources(service):
    try:
        nanny_client = nanny.ServiceRepoClient(
            NANNY_URL,
            secrets.find_nanny_token(),
            attempts=5,
            timeout=1,
            delay=1,
        )
        resources = nanny_client.get_active_runtime_attrs(service)['content']['resources']
        logging.debug("Got resources for %s: %s", service, str(resources))
        return resources
    except Exception as e:
        logging.debug("Could not get resources of nanny service %s, error: %s", service, str(e))
        raise e

NANNY_URL = 'https://nanny.yandex-team.ru'
