import infra.callisto.controllers.sdk as sdk

from location_observer import LocationObserver, LocationState

import commercial_consts
import logging


class LocationObserverCtrl(sdk.Controller):
    observed_services = {}
    files_to_check = ()

    def __init__(self):
        super(LocationObserverCtrl, self).__init__()

        self._location_observers = {
            # 'man': LocationObserver(self.observed_services['man'], self.files_to_check),
            'sas': LocationObserver(self.observed_services['sas'], self.files_to_check),
            'vla': LocationObserver(self.observed_services['vla'], self.files_to_check),
        }

        self._last_known_common_config = None

    def __str__(self):
        return 'Location state observer ({0})'.format(self.path)

    def update(self, reports):
        for observer in self._location_observers.itervalues():
            observer.update()

    def json_view(self):
        json = {
            location: _state_to_json(observer.state)
            for location, observer in self._location_observers.iteritems()
        }
        json['common'] = _state_to_json(LocationState(self.common_state, self.common_config))

        return json

    @property
    def common_state(self):
        state = self._location_observers['sas'].state.yt_state
        for location, observer in self._location_observers.iteritems():
            logging.debug('%s state: %s', location, observer.state.yt_state)
            if state != observer.state.yt_state:
                logging.debug('State %s in sas differs from %s in %s', state, observer.state.yt_state, location)
                return None

        return state

    @property
    def common_config(self):
        config = self._location_observers['sas'].state.config
        for location, observer in self._location_observers.iteritems():
            if config != observer.state.config:
                logging.debug('Config %s in sas differs from %s in %s', config, observer.state.config, location)
                return self._last_known_common_config

        self._last_known_common_config = config

        return config


def _state_to_json(state):
    return {
        'yt_state': state.yt_state,
        'config': {
            'base': state.config.base,
        } if state.config else None
    }


class RIMLocationObserverCtrl(LocationObserverCtrl):
    path = 'rim_oldprod'
    observed_services = {
        # 'man': 'man-images-rim-resources',
        'sas': 'sas-images-rim-resources',
        'vla': 'vla-images-rim-resources',
    }
    files_to_check = ('rim',)


class CommercialLocationObserverCtrl(LocationObserverCtrl):
    path = 'location_observer'
    observed_services = {
        # 'man': commercial_consts.RESOURCES_NANNY_SERVICE_MAN,
        'sas': commercial_consts.RESOURCES_NANNY_SERVICE_SAS,
        'vla': commercial_consts.RESOURCES_NANNY_SERVICE_VLA,
    }
    files_to_check = ('commercialdaemon',)
