import logging

import infra.callisto.controllers.multibeta.generator as generator

import configuration


class BaseBetaGenerator(generator.GeneratorProdTable):
    def _eval_target(self, yappy_conf, prod_conf, configs):
        return configuration.BaseBetaConfiguration(
            yappy_conf['revision'],
            _eval_mmeta(yappy_conf, prod_conf, configs),
            _eval_base(yappy_conf, prod_conf, configs),
            _eval_cbr(yappy_conf, prod_conf, configs),
        )

    def _prod_common_config(self):
        cfg = self._prod_table.last_status()
        return _convert_prod_common_config_to_yappy(cfg.config) if cfg.config else {}


class MmetaBetaGenerator(generator.GeneratorProdTable):
    def _eval_target(self, yappy_conf, prod_conf, configs):
        return configuration.MmetaBetaConfiguration(
            yappy_conf['revision'],
            _eval_mmeta(yappy_conf, prod_conf, configs),
        )

    def _prod_common_config(self):
        cfg = self._prod_table.last_status()
        return _convert_prod_common_config_to_yappy(cfg.config) if cfg.config else {}


class QuickBetaGenerator(generator.GeneratorProdTable):
    def __init__(self, *args, **kwargs):
        super(QuickBetaGenerator, self).__init__(*args, **kwargs)
        self._target_index_state = None

    def _eval_target(self, yappy_conf, prod_conf, configs):
        return configuration.QuickBetaConfiguration(
            yappy_conf['revision'],
            str(self._target_index_state.timestamp),
            _eval_quick_base(yappy_conf, prod_conf, configs),
        )

    def _prod_common_config(self):
        cfg = self._prod_table.last_status()
        return _convert_prod_common_config_to_yappy(cfg.config) if cfg.config else {}

    def set_target_index_state(self, target_state):
        self._target_index_state = target_state

    def execute(self):
        if self._target_index_state:
            super(QuickBetaGenerator, self).execute()
        else:
            logging.warning('Skip execute due to target_index_state is %s', self._target_index_state)


def _eval_mmeta(yappy_conf, prod_conf, configs):
    return configuration.MmetaConfiguration(
        generator.get_resource('mmeta.executable', yappy_conf, prod_conf),
        generator.get_resource('mmeta.models', yappy_conf, prod_conf),
        configs['mmeta.config'],
        configs['mmeta.config_path'],
        yappy_conf['mmeta.rearrange_data'],
        yappy_conf['mmeta.rearrange_index'],
    )


def _eval_base(yappy_conf, prod_conf, configs):
    return configuration.BasesearchConfiguration(
        generator.get_resource('basesearch.executable', yappy_conf, prod_conf),
        generator.get_resource('basesearch.models', yappy_conf, prod_conf),
        configs['basesearch.config'] or get_prod_resource('basesearch.config', prod_conf),
        configs['basesearch.config_path'] if configs['basesearch.config'] else None,
    )


def _eval_cbr(yappy_conf, prod_conf, configs):
    return configuration.CbrsearchConfiguration(
        generator.get_resource('basesearch.executable', yappy_conf, prod_conf),
        configs['cbr.config'] or get_prod_resource('cbr.config', prod_conf),
        configs['cbr.config_path'] if configs['cbr.config'] else None,
    )


def _eval_quick_base(yappy_conf, prod_conf, configs):
    return configuration.RtyserverConfiguration(
        generator.get_resource('quick_base.executable', yappy_conf, prod_conf),
        generator.get_resource('quick_base.models', yappy_conf, prod_conf),
        generator.get_resource('quick_base.static_models', yappy_conf, prod_conf),
        generator.get_resource('quick_base.config', yappy_conf, prod_conf),
        generator.get_resource('quick_base.shardwriter_config', yappy_conf, prod_conf),
    )


def get_prod_resource(resource_name, prod_conf):
    return 'sbr:{}'.format(prod_conf[resource_name]['resource_id'])


def _convert_prod_common_config_to_yappy(cfg):
    quick_base, base, mmeta = cfg['quick_base'], cfg['base'], cfg['mmeta']

    return {
        'quick_base.executable': quick_base['httpsearch'],
        'quick_base.models': quick_base['models.archive'],
        'quick_base.static_models': quick_base['static_models'],
        'quick_base.config': quick_base['config'],
        'quick_base.shardwriter_config': quick_base['shardwriter_config'],

        'basesearch.executable': base['httpsearch'],
        'basesearch.models': base['models.archive'],
        'basesearch.config': base.get('config', 996338681),  # TODO: HARDCODED, due to prod has no common config

        'cbr.executable': base['httpsearch'],
        'cbr.config': base.get('config', 996338710),  # TODO: HARDCODED, due to prod has no common config

        'mmeta.executable': mmeta['httpsearch'],
        'mmeta.models': mmeta['models.archive'],
    }
