import logging
import datetime

import infra.callisto.libraries.yt as yt_utils


class YappyTable(yt_utils.SortedYtTable):
    def __init__(self, yt_client, path, readonly):
        self.schema = [
            {'name': 'time', 'type': 'string', 'sort_order': 'ascending'},
            {'type': 'string', 'name': 'slot_id'},
            {'type': 'string', 'name': 'mmeta.executable'},
            {'type': 'string', 'name': 'mmeta.models'},
            {'type': 'string', 'name': 'mmeta.rearrange_data'},
            {'type': 'string', 'name': 'mmeta.rearrange_index'},
            {'type': 'string', 'name': 'basesearch.models'},
            {'type': 'string', 'name': 'basesearch.executable'},
            {'type': 'string', 'name': 'quick_base.executable'},
            {'type': 'string', 'name': 'quick_base.config'},
            {'type': 'string', 'name': 'quick_base.models'},
            {'type': 'string', 'name': 'quick_base.static_models'},
            {'type': 'string', 'name': 'quick_base.shardwriter_config'},
            {'type': 'int64', 'name': 'revision'},
        ]
        super(YappyTable, self).__init__(yt_client, path, readonly)

    def slot_ids(self):
        request = 'slot_id from [{}] group by slot_id'.format(self.path)
        return [row['slot_id'] for row in self._select_rows(request)]

    def slot_configuration(self, slot_id):
        request = '* from [{}] where slot_id = "{}" order by revision desc limit 1'.format(self.path, slot_id)
        for row in self._select_rows(request):
            return row


class ConfigsTable(yt_utils.SortedYtTable):
    schema = [
        {'name': 'time',                    'type': 'string', 'sort_order': 'ascending'},
        {'name': 'slot_id',                 'type': 'string'},
        {'name': 'mmeta.config',            'type': 'string'},
        {'name': 'mmeta.config_path',       'type': 'string'},
        {'name': 'int.config',              'type': 'string'},
        {'name': 'int.config_path',         'type': 'string'},
        {'name': 'basesearch.config',       'type': 'string'},
        {'name': 'basesearch.config_path',  'type': 'string'},
        {'name': 'cbr.config',              'type': 'string'},
        {'name': 'cbr.config_path',         'type': 'string'},
    ]
    default_config_path = 'all/{short_host}:{port}.cfg'

    def slot_ids(self):
        request = 'slot_id from [{}] group by slot_id'.format(self.path)
        return [row['slot_id'] for row in self._select_rows(request)]

    def slot_configs(self, slot_id):
        request = '* from [{}] where slot_id = "{}" order by time desc limit 1'.format(self.path, slot_id)
        for row in self._select_rows(request):
            return row

    def insert_new_configs(
        self,
        slot_id,
        mmeta_config=None,
        int_config=None,
        basesearch_config=None,
        cbr_config=None,
        mmeta_config_path=default_config_path,
        int_config_path=default_config_path,
        basesearch_config_path=None,
        cbr_config_path=None,
    ):
        self._insert_rows([{
            'time': datetime.datetime.now().isoformat(),
            'slot_id': slot_id,
            'mmeta.config': mmeta_config,
            'mmeta.config_path': mmeta_config_path,
            'int.config': int_config,
            'int.config_path': int_config_path,
            'basesearch.config': basesearch_config,
            'basesearch.config_path': basesearch_config_path,
            'cbr.config': cbr_config,
            'cbr.config_path': cbr_config_path,
        }])

    def update_slot_configs(
        self,
        slot_id,
        mmeta_config=None,
        int_config=None,
        basesearch_config=None,
        cbr_config=None,
        mmeta_config_path=None,
        int_config_path=None,
        basesearch_config_path=None,
        cbr_config_path=None,
    ):
        configs = self.slot_configs(slot_id)
        assert configs
        for key, value in {
            'time': datetime.datetime.now().isoformat(),
            'mmeta.config': mmeta_config,
            'mmeta.config_path': mmeta_config_path,
            'int.config': int_config,
            'int.config_path': int_config_path,
            'basesearch.config': basesearch_config,
            'basesearch.config_path': basesearch_config_path,
            'cbr.config': cbr_config,
            'cbr.config_path': cbr_config_path,
        }.items():
            if value:
                configs[key] = value
        self._insert_rows([configs])


class IndexStateTable(yt_utils.SortedYtTable):
    def __init__(self, yt_client, path):
        super(IndexStateTable, self).__init__(yt_client, path, True)

    def last_state(self):
        request = 'State from [{}] order by Time desc limit 1'.format(self.path)
        for row in self._select_rows(request):
            return row


_log = logging.getLogger(__name__)
