import datetime
import functools
import logging

import infra.callisto.controllers.deployer2.controller as deploy_controller
import infra.callisto.controllers.sdk.notify as notify
import infra.callisto.controllers.sdk.registry as registry
import infra.callisto.controllers.sdk.tier as tier
import infra.callisto.controllers.search_source.controller as search_source
import infra.callisto.controllers.slots as slots
import infra.callisto.controllers.utils.sandbox_utils as sandbox_utils
import infra.callisto.libraries.yt as yt_utils

from infra.callisto.controllers.user.jupiter.chunks.external import Controller as ExternalChunksController
from notifications import PipNotReadyNotification
import sandbox_tasks
import tier_chunks


default_topology = 'stable-154-r2857'
deployer_groups = (
    ('VLA_IMGS_DEPLOY_PIP', default_topology),
)

LOCATION = 'pip'
NAMESPACE_PREFIX = '/images/prod/'
NAMESPACE_PREFIX_YT = '/images/prod/yt'
SUBRESOURCES = ('',)

PIP_BASESEARCH_GROUP = 'VLA_IMGS_BASE_PIP'
PIP_REMOTE_STORAGE_GROUP = 'VLA_IMGS_REMOTE_STORAGE_BASE_PIP'


def slots_(topology=default_topology):
    return (
        slots.Slot('ImgTier0', tier.ImgTier0, PIP_BASESEARCH_GROUP, topology),
        slots.Slot('Embedding', tier.ImgEmbeddingTier0, 'VLA_IMGS_EMBEDDING_PIP', topology, use_mtn=True),
        slots.Slot('InvIndex', tier.ImgInvIndexTier0, 'VLA_IMGS_INVERTED_INDEX_PIP', topology, use_mtn=True),
        slots.Slot('CbrImgTier0', tier.ImgTier0, 'VLA_IMGS_CBIR_BASE_PIP', topology),  # is added as custom tag in nanny service for reports
        slots.Slot('ImgMmetaTier0', tier.ImgMmetaTier0, 'VLA_IMGS_MMETA_PIP', topology),
    )


def get_yt_target_table(readonly):
    yt_client = yt_utils.create_yt_client('arnold', use_rpc=True)
    return search_source.get_yt_target_table(yt_client, 'images/pip/target_v2', readonly)


def get_yt_status_table(readonly):
    yt_client = yt_utils.create_yt_client('arnold', use_rpc=True)
    return search_source.get_yt_status_table(yt_client, '/images/pip/status', readonly)


class PipController(search_source.YtDrivenSourceController):
    def __init__(self, name, slots_, deployer, target_table, status_table, readonly=True):
        super(PipController, self).__init__(name, slots_, deployer, target_table, status_table)
        self.readonly = readonly
        self._notifications = notify.NotificationsAggregator()

    def notifications(self):
        super_notifications = super(PipController, self).notifications()
        since_target_modified = int((datetime.datetime.now() - self._target_modification_time).total_seconds())
        if (
            self._target_modification_time != datetime.datetime.min
            and self._searcher_target_state
            and self._searcher_target_state != self.get_observed_state()[1]
        ):
            return super_notifications + [PipNotReadyNotification(since_target_modified)] + self._notifications.get_notifications()
        return super_notifications + [PipNotReadyNotification(0)] + self._notifications.get_notifications()

    def execute(self):
        super(PipController, self).execute()
        self._notifications.push_buffer()

    def _run_switch_pip_configs_task(self, db_timestamp):
        if not db_timestamp:
            return

        sb_task = sandbox_tasks.switch_pip_configs_task(db_timestamp, self.readonly)
        if not sb_task:
            logging.debug('Switch PIP Task is not created (MODE: %s)', self.readonly)
        elif sandbox_utils.is_task_failed(sb_task):
            logging.debug('There is failed switch PIP task id %s', sb_task['id'])
            self._notifications.add_notification(
                notify.TextNotification(
                    'Check switch PIP task {} and run it manually'.format(sb_task['url']),
                    notify.NotifyLevels.WARNING,
                )
            )
        else:
            logging.debug('Switch PIP task found %s', sb_task['url'])


def make_controller(readonly):
    deploy_ctrl = deploy_controller.make_controller(
        deployer_groups,
        download_args_callback=lambda resource, host: {'max_dl_speed': '100M', 'hardlink': True},
        mtn=True
    )
    slot_ctrls = search_source.make_slot_controllers(slots_(), deploy_ctrl, namespace_prefix=NAMESPACE_PREFIX)
    slot_ctrls['tier0_chunks_ext'] = ExternalChunksController(
        tier_chunks.location_status(LOCATION),
        tier_chunks.location_target(LOCATION),
        tier=tier.ImgTier0,
        optional=True,
        readonly=readonly
    )

    return PipController(
        name='pip_images',
        slots_=slot_ctrls,
        deployer=deploy_ctrl,
        target_table=get_yt_target_table(readonly),
        status_table=get_yt_status_table(readonly=readonly),
        readonly=readonly,
    )


registry.register('images/prod/acceptance', make_controller, [registry.ReportsBackends.V2.vla])
