import collections
import datetime
import logging

import infra.callisto.controllers.build.innerbuilder as builder
import infra.callisto.controllers.sdk.registry as registry
import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.utils.gencfg_api as gencfg_api
import infra.callisto.controllers.sdk.tier as tier
import infra.callisto.controllers.slots.extctrl as extctrl
import infra.callisto.libraries.yt as yt_utils

import rim_common as common

from location_observer_ctrl import RIMLocationObserverCtrl
import rim_pip
import yp_location_ctrl
import yt_observers


builders_topology = 'stable-152-r714'
deploy_units_by_location = {
    'sas': 'sas-imgs-rim.sas-imgs-rim',
    # 'man': 'man-imgs-rim.man-imgs-rim',
    'vla': 'vla-imgs-rim.vla-imgs-rim'
}


class Ctrl(sdk.Controller):
    """
    https://nanny.yandex-team.ru/ui/#/services/catalog/man_imgs_rim_build/"
    """
    path = 'rim'

    def __init__(self, readonly):
        super(Ctrl, self).__init__()
        self.readonly = readonly

        self.builder = builder.make_controller(
            yt_observers.ImgRIMTier,
            instance_provider=gencfg_api.InstanceProvider(
                [gencfg_api.GencfgGroup('VLA_IMGS_RIM_3K_BUILD', builders_topology, mtn=True)],
                report_tags={'VLA_IMGS_RIM_3K_BUILD'}
            ),
            space_needed_full_build_coeff=1.0,
            space_needed_inc_build_coeff=0.3,
            assist_last_n_on_finish=300,
        )

        self._prod_ruler = make_ruler()
        self.prod = RIMLocationObserverCtrl()

        _ctrls = 'http://ctrl.clusterstate.yandex-team.ru'
        self.pip = extctrl.ExternalSlotsController(_ctrls + '/rim/pip', readonly, rim_pip.get_yt_target_table(readonly))
        # self.man_rim = extctrl.ExternalSlotsController(_ctrls + '/rim/man', readonly, common.get_yt_target_table('man', readonly))
        self.sas_rim = extctrl.ExternalSlotsController(_ctrls + '/rim/sas', readonly, common.get_yt_target_table('sas', readonly))
        self.vla_rim = extctrl.ExternalSlotsController(_ctrls + '/rim/vla', readonly, common.get_yt_target_table('vla', readonly))

        self.register(
            self.prod,
            self.builder,
            self.pip,
            # self.man_rim,
            self.sas_rim,
            self.vla_rim,
        )

    def execute(self):
        if self.prod.common_state:
            prod_timestamp = int(self.prod.common_state)

            self._set_pip(prod_timestamp)

            if datetime.datetime.now().hour in (8, 20):
                self._set_prod(prod_timestamp)
        else:
            logging.debug('No common state observed, leaving pip & prod intact')

    def _set_pip(self, prod_timestamp):
        deploy_target = self._prod_ruler.to_deploy_on_acceptance(prod_timestamp)
        basesearch_target = self._prod_ruler.search_target_on_acceptance(deploy_target, prod_timestamp)
        self.pip.set_target_state(deploy_target, basesearch_target, null_if_not_deployed=True)

    def _set_prod(self, prod_timestamp):
        deploy_target = self._prod_ruler.to_deploy_on_prod(prod_timestamp)
        assert prod_timestamp in deploy_target

        for index, location in enumerate((self.sas_rim, self.vla_rim)):
            if index * 15 < datetime.datetime.now().minute < (index + 1) * 15:
                location.set_target_state(deploy_target, prod_timestamp)

    def json_view(self):
        result = collections.OrderedDict()
        result['__readonly__'] = self.readonly
        # result['man_rim'] = self.man_rim.json_view()
        result['sas_rim'] = self.sas_rim.json_view()
        result['vla_rim'] = self.vla_rim.json_view()
        result['builders'] = {
            self.builder.tier.name: self.builder.json_view()
        }
        result['prod'] = self.prod.json_view()

        return result

    def build_progress(self, tier_name):
        return self.builder.build_progress(tier_name)


def make_ruler():
    acceptance = sdk.prod_ruler.AcceptanceTableSource(
        path='//home/images/relatedimages/shard_deploy/acceptance_result',
        yt_client=yt_utils.create_yt_client('arnold')
    )
    return sdk.prod_ruler.DeployRuler(
        build_source=yt_observers.ImgRIMTier.source,
        acceptance_source=acceptance,
        max_states_to_keep=2,
    )


def make_ctrl(readonly):
    return Ctrl(readonly)


def make_yp_location_ctrl_prod_params(location, readonly):
    return yp_location_ctrl.MakeYpControllerParams(
        name='{}_rim'.format(location),
        location=location,
        deploy_unit_id=deploy_units_by_location[location],
        slot_name='Rim',
        tier=tier.ImgRIMTier,
        namespace='/images/prod/rim/',
        target_table=common.get_yt_target_table(location, readonly),
        is_pip=False,
    )


def make_sas_controller(readonly):
    return yp_location_ctrl.make_yp_controller(make_yp_location_ctrl_prod_params('sas', readonly))


def make_man_controller(readonly):
    return yp_location_ctrl.make_yp_controller(make_yp_location_ctrl_prod_params('man', readonly))


def make_vla_controller(readonly):
    return yp_location_ctrl.make_yp_controller(make_yp_location_ctrl_prod_params('vla', readonly))


registry.register('images/prod/rim', make_ctrl, [registry.ReportsBackends.V2.vla])
registry.register('images/prod/rim/sas', make_sas_controller, [registry.ReportsBackends.V2.sas])
# registry.register('images/prod/rim/man', make_man_controller, [registry.ReportsBackends.V2.man])
registry.register('images/prod/rim/vla', make_vla_controller, [registry.ReportsBackends.V2.vla])
