import infra.callisto.controllers.deployer2.controller as deploy_controller
import infra.callisto.controllers.search_source.controller as search_source
import infra.callisto.controllers.slots as slots
import infra.callisto.controllers.utils.entities as entities
import infra.callisto.controllers.utils.yp_utils as yp_utils

from simple_pip_ctrl import SimplePipController


yp_masters_by_location = {
    'sas': yp_utils.YpMasters.sas,
    # 'man': yp_utils.YpMasters.man,
    'vla': yp_utils.YpMasters.vla
}


class DownloadArgsCallback(object):
    def __init__(self, deployer_instances):
        self._hdd_only_hosts = {
            agent.host for agent, spec in deployer_instances.items()
            if 'itag_copy_on_ssd' not in spec['tags']
        }

    def __call__(self, resource, host):
        if host in self._hdd_only_hosts:
            return {'max_dl_speed': '15M'}
        return {'max_dl_speed': '50M'}


class MakeYpControllerParams(object):
    def __init__(self, name, location, deploy_unit_id, slot_name, tier, namespace,
                 target_table, status_table=None, is_pip=False, deployer_port=10000, base_port=1080):
        self.name = name
        self.location = location
        self.deploy_unit_id = deploy_unit_id
        self.slot_name = slot_name
        self.tier = tier
        self.namespace = namespace
        self.target_table = target_table
        self.status_table = status_table
        self.is_pip = is_pip
        self.deployer_port = deployer_port
        self.base_port = base_port


def make_deployer_controller(pods, deploy_unit_id, deployer_port):
    return deploy_controller.DeployerController(
        {pod.fqdn: entities.Agent(pod.fqdn, deployer_port) for pod in pods},
        download_args_callback=DownloadArgsCallback({}),
        tags={'a_itype_deployer', deploy_unit_id}
    )


def make_yp_controller(params):
    pods = yp_utils.read_pods(yp_masters_by_location[params.location], params.deploy_unit_id)
    deployer_ctrl = make_deployer_controller(pods, params.deploy_unit_id, params.deployer_port)

    slot_ctrls = {params.slot_name: slots.make_slot_controller(
        slots.Slot(params.slot_name, params.tier, None, None),
        deployer_ctrl,
        yp_utils.agent_shard_number_mapping(pods, params.base_port, params.tier),
        deploy_only=True,
        namespace_prefix=params.namespace,
    )}

    if params.is_pip:
        ctrl_cls = SimplePipController
    else:
        ctrl_cls = search_source.YtDrivenSourceController

    return ctrl_cls(
        name=params.name,
        slots_=slot_ctrls,
        deployer=deployer_ctrl,
        target_table=params.target_table,
        status_table=params.status_table
    )
