import datetime
import logging
import pytz

import yt.wrapper as yt
from yt.yson import YsonUint64


from infra.callisto.deploy.tracker.core.table import NamespaceTable, namespace_to_path
import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.sdk.tier as tiers

from search.plutonium.deploy.chunks.proto.chunks_pb2 import TChunkCtlResourceSpec
import search.base_search.common.protos.docs_tier_pb2 as docs_tier
import search.base.blob_storage.config.protos.chunk_access_type_pb2 as access_type

import tier_chunks

_log = logging.getLogger(__name__)

MOSCOW_TZ = pytz.timezone('Europe/Moscow')

IMAGESINDEX_MRPREFIX = '//home/images/index'
SPEC_PARTS = '//home/cajuper/user/images/prod/chunks/ctl/spec_parts'
SPECS_TABLE = '//home/cajuper/user/images/prod/chunks/ctl/specs'


def prepare_images_remote_storage_torrents(timestamp, target, tier, yt_client=yt):
    merged_inputs = ['{}/{}/{}'.format(IMAGESINDEX_MRPREFIX, timestamp_to_index_state(timestamp), 'shards/archive.rs.merged_torrents')]

    now = datetime.datetime.now().isoformat()

    def reducer(key, rows):
        for row in rows:
            shard_number, oth = key['Name'].split('/', 1)

            name = '{}/{}'.format(tier.make_shard(shard_number, timestamp), oth)
            yield {
                'name': name,
                'rbtorrent': row['Rbtorrent'],
                'time': now,
                'size': 0
            }
    yt_client.run_reduce(reducer, merged_inputs, yt.TablePath(target, schema=NamespaceTable.schema), reduce_by='Name')


def ensure_path(yt_client, path):
    if not yt_client.exists(path):
        ensure_path(yt_client, path.rsplit('/', 1)[0])
        yt_client.create('map_node', path)


def make_dynamic(yt_client, path):
    if yt_client.exists(path):
        if not yt_client.get(path + '/@dynamic'):
            yt_client.alter_table(path, dynamic=True)
        if yt_client.get(path + '/@tablet_state') == 'unmounted':
            yt_client.set(path + '/@tablet_cell_bundle', 'cajuper')
            yt_client.set(path + '/@primary_medium', 'ssd_blobs')
            yt_client.mount_table(path)


def timestamp_to_index_state(ts):
    return datetime.datetime.fromtimestamp(int(ts), tz=MOSCOW_TZ).strftime('%Y%m%d-%H%M%S')


def prepare_chunk_ctl_specs(timestamp, yt_client):
    merged_inputs = '{}/{}/{}'.format(IMAGESINDEX_MRPREFIX, timestamp_to_index_state(timestamp), 'shards/archive.rs.merged_torrents')
    ensure_path(yt_client, SPEC_PARTS)
    spec_tmp_part = SPEC_PARTS + '/' + str(timestamp) + '.tmp'
    spec_merged_marker = SPEC_PARTS + '/' + str(timestamp) + '.merged'

    if yt_client.exists(spec_merged_marker):
        return

    def reducer(key, rows):
        shard, oth = key['Name'].split('/', 1)

        spec = TChunkCtlResourceSpec()
        spec.Labels.Label["AccessType"] = str(access_type.Disk)
        generator = tier_chunks.RemoteStorageChunksGenerator()
        spec.Labels.Label["Tier"] = str(docs_tier.ImgTier0)

        for row in rows:
            part = spec.Source.Compound.Sources.add()
            part.Source.Skynet.Url = row['Rbtorrent']
            part.Source.Skynet.Size = int(generator.chunks_size)

        spec.Labels.Label["Shard"] = shard
        spec.Labels.Label["Version"] = str(timestamp)
        spec.QuotaId = str(timestamp)

        shard_name = tiers.ImgTier0.prefix + '-' + str(shard)

        def make_row(shard, timestamp, chunk, part, spec):
            return {
                'Stream': shard,
                'Snapshot': str(timestamp),
                'Table': '',
                'ChunkId': YsonUint64(chunk),
                'PartId': YsonUint64(part),
                'Resource': spec.SerializeToString()
            }

        if 'remote_storage' in oth:  # lrc
            chunk, part = map(int, oth.split('/')[1:])
            spec.Labels.Label["ChunkId"] = str(chunk)
            spec.Labels.Label["PartId"] = str(part)

            spec.Replication = 1
            spec.MinAliveReplica = 1

            if part < 12:
                spec.BalancingWeight = 1
            elif part < 14:
                spec.BalancingWeight = 0.5
            else:
                spec.BalancingWeight = 0.1
            yield make_row(shard_name, timestamp, chunk, part, spec)
        elif 'multi_level_cache_replicas' in oth:  # replicas
            chunk, part = map(int, oth.split('/')[1:])
            spec.Labels.Label["ChunkId"] = str(chunk)
            spec.Replication = generator.replicas_parts_replication
            spec.MinAliveReplica = generator.replicas_parts_replication - 1
            spec.BalancingWeight = 0.001
            assert part == 0
            yield make_row(shard_name, timestamp, chunk, part, spec)

    yt_client.run_reduce(reducer, merged_inputs, spec_tmp_part, reduce_by='Name')
    yt_client.run_sort(spec_tmp_part, "<append=%true>" + SPECS_TABLE, sort_by=['Stream', 'Snapshot', 'Table', 'ChunkId', 'PartId'])
    yt_client.remove(spec_tmp_part)
    yt_client.create('map_node', spec_merged_marker)


class ImagesChunkCtlSpecsController(sdk.Controller):
    def __init__(self, observer, yt_client, readonly):
        self._observer = observer
        self._yt_client = yt_client
        super(ImagesChunkCtlSpecsController, self).__init__()
        self._readonly = readonly

    def execute(self):
        if not self._readonly:
            for generation in self._observer.get_last_generations(2):
                prepare_chunk_ctl_specs(generation, self._yt_client)

    def __str__(self):
        return '{}({})'.format(self.__class__.__name__, self._observer.tier.name)


class ImagesCopierController(sdk.Controller):
    def __init__(self, observer, namespace_prefix, yt_client, readonly):
        self._observer = observer
        self._yt_client = yt_client
        self._namespace_prefix = namespace_prefix
        sdk.Controller.__init__(self)
        self._readonly = readonly

    def __str__(self):
        return '{}({})'.format(self.__class__.__name__, self._observer.tier.name)

    def _prepare_torrents(self, timestamp, output_path, yt_client):
        raise NotImplementedError()

    def execute(self):
        for generation in self._observer.get_last_generations(2):
            namespace = '{}/{}/{}'.format(self._namespace_prefix, generation, self._observer.tier.name)
            path = namespace_to_path(namespace)
            if not self._readonly:
                ensure_path(self._yt_client, path.rsplit('/', 1)[0])
            if not self._yt_client.exists(path):
                try:
                    _log.info('create namespace for yt copier %s', namespace)
                    if self._readonly:
                        continue
                    self._prepare_torrents(generation, path, self._yt_client)
                except Exception as e:
                    _log.exception(e)
            if not self._readonly:
                make_dynamic(self._yt_client, path)


class ImagesChunksCopierController(ImagesCopierController):
    def __init__(self, observer, namespace_prefix, yt_client, readonly):
        ImagesCopierController.__init__(self, observer, namespace_prefix, yt_client, readonly)

    def _prepare_torrents(self, timestamp, output_path, yt_client):
        prepare_images_remote_storage_torrents(timestamp, output_path, self._observer.tier, yt_client)
