import collections
import infra.callisto.controllers.utils.entities as entities

import table
import planner

from planner import HostSpec, ResourceSpec


class Allocator(object):
    def __init__(self, location, tier, chunk_namespace, readonly=True, **generator_kwargs):
        self._location = location
        self._tier = tier
        self._readonly = readonly
        self._chunk_namespace = chunk_namespace
        self._generator_kwargs = generator_kwargs
        table.ensure_path(self._namespace, 'mapping')

    def mapping_exists(self, generation):
        return table.mapping_exists(self._namespace, generation)

    def ensure_mapping_removed(self, generation):
        if not self._readonly:
            return table.ensure_mapping_removed(self._namespace, generation)

    def load_mapping(self, generation):
        return table.get_mapping_table(self._namespace, generation).load()

    def generate_mapping(self, generation, chunks_specs, hosts_specs):
        mapping = _generate_mapping(generation, chunks_specs, hosts_specs, **self._generator_kwargs)
        if not self._readonly:
            table.get_mapping_table(self._namespace, generation, readonly=False).dump(mapping)
        return mapping

    @property
    def tier(self):
        return self._tier

    @property
    def _namespace(self):
        return self._chunk_namespace + '/' + self._location + '/' + self._tier.name


def _convert_from_timeless(generation, mapping):
    return {
        _substitute_timestamp(generation, timeless_chunk): hosts
        for timeless_chunk, hosts in mapping.iteritems()
    }


def _drop_timestamp(chunk):
    return _TimelessChunk(chunk.shard.tier, chunk.shard.number, chunk.path)


def _substitute_timestamp(generation, timeless_chunk):
    return entities.Chunk(
        timeless_chunk.tier.make_shard(timeless_chunk.shard_number, generation),
        timeless_chunk.path
    )


def _generate_mapping(generation, chunks, hosts, **kwargs):
    resources = [ResourceSpec(_drop_timestamp(spec.resource), spec.size, spec.replication, spec.parts_count) for spec in chunks]
    return _convert_from_timeless(generation, planner.consistent_mapping(resources, hosts, **kwargs))


_TimelessChunk = collections.namedtuple('_TimelessChunk', ['tier', 'shard_number', 'path'])

__all__ = [
    'Allocator',
    'HostSpec',
    'ResourceSpec',
]
