import collections

import infra.callisto.controllers.sdk.tier as tiers
import infra.callisto.libraries.yt as yt_utils
import infra.callisto.controllers.utils.entities as entities
import infra.callisto.deploy.resource as deploy_resource


PATH_PREFIX = '//home/cajuper/user'


def _path(path_prefix, table=None, generation=None):
    path = deploy_resource.safe_join_path(PATH_PREFIX, path_prefix)
    if table:
        path = deploy_resource.safe_join_path(path, table)
    if generation:
        path = deploy_resource.safe_join_path(path, str(generation))
    return path


def _get_yt_client(use_rpc=False):
    return yt_utils.create_yt_client('arnold', use_rpc=use_rpc)


def get_mapping_table(path_prefix, base_generation, readonly=True):
    return MappingTable(
        _get_yt_client(),
        _path(path_prefix, 'mapping', base_generation),
        readonly,
    )


def ensure_path(path_prefix, table):
    client = _get_yt_client()
    if not client.exists(_path(path_prefix, table)):
        client.create('map_node', _path(path_prefix, table), recursive=True)


def list_generations(path_prefix):
    return map(int, _get_yt_client().list(_path(path_prefix, 'mapping')))


def table_generated(path):
    client = _get_yt_client()
    return client.exists(path) and client.has_attribute(path, '_finished')


def ensure_table_removed(path):
    client = _get_yt_client()
    if client.exists(path):
        client.remove(path, force=True)


def mapping_exists(path_prefix, generation):
    return table_generated(_path(path_prefix, 'mapping', generation))


def ensure_mapping_removed(path_prefix, generation):
    return ensure_table_removed(_path(path_prefix, 'mapping', generation))


class MappingTable(yt_utils.SortedYtTable):
    schema = [
        {'name': 'shard',   'type': 'string', 'sort_order': 'ascending'},
        {'name': 'path',    'type': 'string', 'sort_order': 'ascending'},
        {'name': 'host',    'type': 'string', 'sort_order': 'ascending'},
        {'name': 'void',    'type': 'boolean'}
    ]

    def load(self):
        result = collections.defaultdict(set)
        for row in self._select_rows(
                request='* from [{}]'.format(self._path),
                input_row_limit=5000 * 1000,
                output_row_limit=5000 * 1000,
        ):
            chunk = entities.Chunk(shard=tiers.parse_shard(row['shard']), path=row['path'])
            result[chunk].add(row['host'])
        assert result
        return dict(result)

    def dump(self, chunk_hosts_mapping):
        rows = []
        for chunk, hosts in chunk_hosts_mapping.iteritems():
            for host in hosts:
                rows.append({
                    'shard': chunk.shard.fullname,
                    'path': chunk.path,
                    'host': host,
                    'void': False,
                })

        try:
            self._insert_many_rows(rows)
            _get_yt_client().set_attribute(self._path, '_finished', True)
        except Exception:
            _get_yt_client().remove(self._path, force=True)
            raise
