import copy

import search.base.blob_storage.config.protos.remote_chunked_blob_storage_index_config_pb2 as remote_index_pb
import search.base.blob_storage.config.protos.remote_chunked_blob_storage_config_pb2 as remote_pb2
import search.base_search.common.protos.docs_tier_pb2 as docs_tier_pb2
import search.base_search.common.protos.source_replics_pb2 as source_replics_pb2

import infra.callisto.deploy.storage.storage as storage
import infra.callisto.controllers.utils.funcs as funcs
import infra.callisto.controllers.sdk as sdk


_HAMSTER_PORT = 8800
_PROD_PORT = 80
_RS_CACHE_PORT_OFFSET = 3
_PROD_UNIXSOCKET_PATH = './remote_storage_cache.sock'
_HAMSTER_UNIXSOCKET_PATH = './hamster_remote_storage_cache.sock'
_CHUNKS_CACHE_SOURCE_NUM = 0
_HAMSTER_TIMEOUTS_MULTIPLICATOR=10


def _remote_storage_url(host, port):
    return 'http://{}:{}/remote_storage@1@1@1@1@1'.format(host, port)


def _remote_storage_cache_url(unixsocket_path=None, host=None, port=80, endpoint_set=None):
    if unixsocket_path is not None:
        return 'http+unix://[{}]/remote_storage@1@0@0@0@0'.format(unixsocket_path)
    elif host is not None:
        return 'http://{}:{}/remote_storage@0@1@1@1@1'.format(host, port)
    elif endpoint_set is not None:
        return 'sd://{}/remote_storage@0@1@1@1@1'.format(endpoint_set)


def _shards_configs(part_hosts_mapping, host_agent_mapping, is_hamster=False,
                    docs_tier=docs_tier_pb2.WebTier1,
                    chunk_count=72,  # ErasureChunksGenerator.erasure_chunks_cnt
                    erasure_chunk_ids=None,  # ErasureChunksGenerator.erasure_chunks_ids()
                    erasure_part_count=16,  # ErasureChunksGenerator.erasure_parts_cnt
                    cache_chunk_ids=None,  # ErasureChunksGenerator.cache_chunks_ids()
                    cache_part_count=1,  # ErasureChunksGenerator.cache_parts_cnt
                    replicas_chunk_ids=None,  # ErasureChunksGenerator.replicas_chunks_ids()
                    replicas_part_count=1,  # ErasureChunksGenerator.replicas_parts_cnt
                    timeouts_multiplicator=1):
    result = {}

    rs_port = _HAMSTER_PORT if is_hamster else _PROD_PORT
    unixsocket_path = _HAMSTER_UNIXSOCKET_PATH if is_hamster else _PROD_UNIXSOCKET_PATH
    erasure_chunk_ids = erasure_chunk_ids or []
    cache_chunk_ids = cache_chunk_ids or []
    replicas_chunk_ids = replicas_chunk_ids or []

    assert not erasure_chunk_ids or len(erasure_chunk_ids) == chunk_count
    assert not cache_chunk_ids or len(cache_chunk_ids) == chunk_count
    assert not replicas_chunk_ids or len(replicas_chunk_ids) == chunk_count

    for part, hosts in part_hosts_mapping.iteritems():
        shard_number = part.shard.shard_number

        if shard_number not in result:
            result[shard_number] = remote_pb2.TRemoteChunkedBlobStorageConfig(
                Tier=docs_tier,
                Shard=shard_number,
            )

            first_cache_chunk = min(cache_chunk_ids or [0])
            count_cache_chunks = len(cache_chunk_ids)
            if count_cache_chunks and unixsocket_path:  # MultiLevelCache CACHE
                result[shard_number].ChunksCache.CopyFrom(_make_chunks_cache(first_cache_chunk, count_cache_chunks, unixsocket_path))

        rs_proto = result[shard_number]

        if erasure_chunk_ids and part.number in erasure_chunk_ids:  # MultiLevelCache ERASURE
            if not rs_proto.HasField('ChunksErasure'):
                rs_proto.ChunksErasure.CopyFrom(_make_chunks_erasure(erasure_chunk_ids, erasure_part_count))

            _add_part_to_erasure_chunk(
                rs_proto,
                part.number % chunk_count,
                part.part,
                {host_agent_mapping[host].host for host in hosts},
                rs_port
            )
        elif replicas_chunk_ids and part.number in replicas_chunk_ids:  # MultiLevelCache REPLICAS
            if not rs_proto.HasField('ChunksReplicas'):
                rs_proto.ChunksReplicas.CopyFrom(_make_chunks_replicas(replicas_chunk_ids, replicas_part_count))

            _add_part_to_replicas_chunk(
                rs_proto,
                part.number % chunk_count,
                part.part,
                {host_agent_mapping[host].host for host in hosts},
                rs_port
            )

    for shard_number in result:
        _add_basesearch_timeouts_options(result[shard_number], is_hamster, timeouts_multiplicator)

    return result


def _make_empty_parts(parts_count):
    return [
        source_replics_pb2.TSourceReplics()
        for _ in xrange(parts_count)
    ]


def _make_empty_chunks(chunk_ids, parts_count):
    return [
        remote_pb2.TChunkSource(
            Chunk=chunk_id,
            Parts=_make_empty_parts(parts_count)
        )
        for chunk_id in chunk_ids
    ]


def _make_chunks_erasure(chunk_ids, parts_count):
    return remote_pb2.TErasureChunks(
        StatName="",
        Chunks=_make_empty_chunks(chunk_ids, parts_count),
    )


def _make_chunks_replicas(chunk_ids, parts_count):
    return remote_pb2.TReplicasChunks(
        StatName="replicas",
        Chunks=_make_empty_chunks(chunk_ids, parts_count),
    )


def _make_chunks_cache(first_chunk, chunk_count, unixsocket_path):
    return remote_pb2.TCacheChunks(
        StatName="cache",
        FirstChunk=first_chunk,
        ChunkCount=chunk_count,
        ChunkReplics=source_replics_pb2.TSourceReplics(
            SourceNum=_CHUNKS_CACHE_SOURCE_NUM,
            SearchScriptCgis=[_remote_storage_cache_url(unixsocket_path=unixsocket_path)],
        ),
    )


def _add_part_to_erasure_chunk(rs_proto, chunk_idx, part_number, hosts, port):
    rs_proto.ChunksErasure.Chunks[chunk_idx].Parts[part_number].SearchScriptCgis.extend([
        _remote_storage_url(host, port)
        for host in hosts
    ])


def _add_part_to_replicas_chunk(rs_proto, chunk_idx, part_number, hosts, port):
    rs_proto.ChunksReplicas.Chunks[chunk_idx].Parts[part_number].SearchScriptCgis.extend([
        _remote_storage_url(host, port)
        for host in hosts
    ])


def _add_replics_to_to_cache_chunks(rs_proto, agent_cache_replicas=None, endpoint_set=None):
    assert agent_cache_replicas is not None or endpoint_set is not None

    if agent_cache_replicas is not None:
        rs_proto.ChunksCache.ChunkReplics.SearchScriptCgis.extend([
            _remote_storage_cache_url(host=x.host, port=x.port + _RS_CACHE_PORT_OFFSET)
            for x in agent_cache_replicas
        ])
    elif endpoint_set is not None:
        rs_proto.ChunksCache.ChunkReplics.SearchScriptCgis.append(_remote_storage_cache_url(endpoint_set=endpoint_set))


def _add_basesearch_timeouts_options(cfg, is_hamster=False, multiplicator=1):
    def ms(milliseconds):
        return "{}ms".format(int(milliseconds * multiplicator))

    # ChunksErasure

    cfg.ChunksErasure.SourceOptions.AllowBalancerDynamic = False

    cfg.ChunksErasure.SourceOptions.TaskOptions.ConnectTimeouts.extend([ms(10)])
    cfg.ChunksErasure.SourceOptions.TaskOptions.SendingTimeouts.extend([ms(10)])
    cfg.ChunksErasure.SourceOptions.BalancingOptions.EnableIpV6 = True
    cfg.ChunksErasure.SourceOptions.BalancingOptions.MaxAttempts = 2
    cfg.ChunksErasure.SourceOptions.BalancingOptions.PingZeroWeight = False
    cfg.ChunksErasure.SourceOptions.BalancingOptions.RandomGroupSelection = True
    cfg.ChunksErasure.SourceOptions.BalancingOptions.AllowDynamicWeights = False
    cfg.ChunksErasure.SourceOptions.BalancingOptions.EnableUnresolvedHosts = True
    cfg.ChunksErasure.SourceOptions.HedgedRequestOptions.HedgedRequestRateThreshold = 0.2
    cfg.ChunksErasure.SourceOptions.HedgedRequestOptions.HedgedRequestTimeouts.extend([ms(5)])
    cfg.ChunksErasure.SourceOptions.TimeOut = ms(40)
    cfg.ChunksErasure.SourceOptions.AllowConnStat = True
    cfg.ChunksErasure.SourceOptions.ConnStatFailThreshold = ms(10)
    cfg.ChunksErasure.SourceOptions.PessimizeAllErrors = True

    cfg.ChunksErasure.ConnStatOptions.FailThreshold = 50
    cfg.ChunksErasure.ConnStatOptions.CheckTimeout = "2s"
    cfg.ChunksErasure.ConnStatOptions.CheckInterval = 1000

    cfg.ChunksErasure.AuxRequestThresholds.extend([7, 16])

    # 1+6x+10x^2 == 2.125
    if is_hamster:
        cfg.ChunksErasure.AuxRequestOptions.WeightedThresholds.extend([1, 1])
    else:
        cfg.ChunksErasure.AuxRequestOptions.WeightedThresholds.extend([0.15, 0.15**2])

    cfg.ChunksErasure.AuxRequestOptions.HedgedRequestTimeouts.extend([ms(8), ms(16)])
    cfg.ChunksErasure.RecoverTimeout = ms(55)

    cfg.ChunksErasure.EndpointSetOptions.UseUnistat = False

    # ChunksCache

    if cfg.HasField('ChunksCache'):
        cfg.ChunksCache.SourceOptions.CopyFrom(cfg.ChunksErasure.SourceOptions)
        cfg.ChunksCache.ConnStatOptions.CopyFrom(cfg.ChunksErasure.ConnStatOptions)

        cfg.ChunksCache.EndpointSetOptions.UseUnistat = False
        cfg.ChunksCache.EndpointSetOptions.OffsetPort = _RS_CACHE_PORT_OFFSET

    # ChunksReplicas

    if cfg.HasField('ChunksReplicas'):
        cfg.ChunksReplicas.SourceOptions.CopyFrom(cfg.ChunksErasure.SourceOptions)
        cfg.ChunksReplicas.SourceOptions.BalancingOptions.MaxAttempts = 3
        cfg.ChunksReplicas.ConnStatOptions.CopyFrom(cfg.ChunksErasure.ConnStatOptions)

    # SdConfig

    cfg.SdConfig.Host = "sd.yandex.net"
    cfg.SdConfig.Port = 8080
    cfg.SdConfig.UpdateFrequency = "60s"
    cfg.SdConfig.ConnectTimeout = "100ms"
    cfg.SdConfig.RequestTimeout = "1s"
    cfg.SdConfig.ClientName = _get_sd_client_name(is_hamster)
    cfg.SdConfig.CacheDir = "sd_cache"
    cfg.SdConfig.Log = "sd_log"


def _get_sd_client_name(is_hamster):
    if is_hamster:
        return "web-basesearch-hamster-multibeta"
    return "web-basesearch-prod"


def generate_basesearch_configs(basesearch_agent_shard_number_map, endpoint_set_template, configs):
    shard_number_agent_map = {}
    for agent, (_, shard_number) in basesearch_agent_shard_number_map.items():
        if shard_number not in shard_number_agent_map:
            shard_number_agent_map[shard_number] = []
        shard_number_agent_map[shard_number].append(agent)

    basesearch_configs = {}
    for agent, (_, shard_number) in basesearch_agent_shard_number_map.items():
        if shard_number not in configs:
            continue
        agent_config = copy.deepcopy(configs[shard_number])

        if agent_config.HasField('ChunksCache'):
            if endpoint_set_template is None:
                agent_cache_replicas = [x for x in shard_number_agent_map[shard_number] if x != agent]
                _add_replics_to_to_cache_chunks(agent_config, agent_cache_replicas=agent_cache_replicas)
            else:
                endpoint_set = endpoint_set_template.format(shard_number)
                _add_replics_to_to_cache_chunks(agent_config, endpoint_set=endpoint_set)

        basesearch_configs[agent] = str(agent_config)

    return basesearch_configs


filename_mapping = {
    'arc': {'arc': 'wad.part'},
    'extarc': {'extarc': 'extinfo.part'},
    'keyinv': {'keyinv': 'keyinv.part'},
    'video': {'arc': 'wad.part', 'extarc': 'extinfo.part', 's2t': 's2t.part', 's2tdssm': 's2tdssm.part', 'omni': 'omni.part'},
    'pudge': {'arc': 'wad.part', 'keyinv': 'keyinv.part'},
    'dynamic': {'extarc': 'extinfo.part', 'common': 'common.part'},
    '': {'arc': 'wad.part', 'extarc': 'extinfo.part', 'keyinv': 'keyinv.part', 'common': 'common.part'}
}


def _rel_path_to_chunk(chunk, namespace_prefix, subresource, filename):
    chunk_resource = sdk.resource.chunk_to_resource(namespace_prefix, chunk, subresource)
    return storage.resource_path('', chunk_resource.namespace, chunk_resource.name) + '/' + filename


def generate_remote_storage_configs(
    tier,
    remote_storage_host_agent_map,
    chunk_hosts_mapping,
    namespace_prefix,
    subresources
):
    result = {}
    host_chunks_mapping = funcs.invert_mapping(chunk_hosts_mapping)
    for agent in remote_storage_host_agent_map.itervalues():
        cfg = remote_index_pb.TRemoteBlobStorageIndexConfig()
        for subresource in subresources:
            for chunk in sorted(host_chunks_mapping.get(agent.node_name, ())):
                cfg.Chunks.extend(_make_subsource_chunks(tier, chunk, namespace_prefix, subresource))
        result[agent] = str(cfg)

    return result


def _make_subsource_chunks(tier, chunk, namespace_prefix, subresource):
    return [
        _make_chunk_proto(tier, chunk, namespace_prefix, subresource, index, filename)
        for index, filename in filename_mapping[subresource].iteritems()
    ]


def _make_chunk_proto(tier, chunk, namespace_prefix, subresource, index, filename):
    return remote_index_pb.TRemoteBlobStorageChunkConfig(
        Tier=tier,
        Shard=chunk.shard.shard_number,
        Version=chunk.shard.timestamp,
        Path=_rel_path_to_chunk(chunk, namespace_prefix, subresource, filename),
        Index=index,
        Id=chunk.number,
        Part=chunk.part,
    )


def _tier_to_docs_tier(tier):
    if tier.name == 'WebTier0':
        return docs_tier_pb2.WebTier0
    elif tier.name == 'WebTier1':
        return docs_tier_pb2.WebTier1
    elif tier.name == 'VideoPlatinum':
        return docs_tier_pb2.VideoPlatinum
    elif tier.name == 'ImgTier0':
        return docs_tier_pb2.ImgTier0

    raise RuntimeError('Tier {} not supported'.format(tier.name))


def generate_configs(
    remote_storage_host_agent_map,
    basesearch_agent_shard_number_map,
    basesearch_hamster_agent_shard_number_map,
    chunk_hosts_mapping,
    chunk_generator,
    namespace_prefix,
    subresources,
    endpoint_set_template,
    endpoint_set_hamster_template,
    timeouts_multiplicator=1,
):
    if not basesearch_agent_shard_number_map and not basesearch_hamster_agent_shard_number_map:
        raise RuntimeError('Neither basesearch_agent_shard_number_map nor basesearch_hamster_agent_shard_number_map were specified')

    tier = _tier_to_docs_tier(chunk_generator.tier)

    configs = {}
    if basesearch_agent_shard_number_map:
        prod_shards_configs = _shards_configs(
            chunk_hosts_mapping, remote_storage_host_agent_map, is_hamster=False,
            docs_tier=tier, chunk_count=chunk_generator.erasure_chunks_cnt,
            erasure_chunk_ids=chunk_generator.erasure_chunk_ids(), erasure_part_count=chunk_generator.erasure_parts_cnt,
            cache_chunk_ids=chunk_generator.cache_chunk_ids(), cache_part_count=chunk_generator.cache_parts_cnt,
            replicas_chunk_ids=chunk_generator.replicas_chunk_ids(), replicas_part_count=chunk_generator.replicas_parts_cnt,
            timeouts_multiplicator=timeouts_multiplicator,
        )
        configs.update(
            generate_basesearch_configs(basesearch_agent_shard_number_map, endpoint_set_template, prod_shards_configs)
        )

    if basesearch_hamster_agent_shard_number_map:
        hamster_shard_configs = _shards_configs(
            chunk_hosts_mapping, remote_storage_host_agent_map, is_hamster=True,
            docs_tier=tier, chunk_count=chunk_generator.erasure_chunks_cnt,
            erasure_chunk_ids=chunk_generator.erasure_chunk_ids(), erasure_part_count=chunk_generator.erasure_parts_cnt,
            cache_chunk_ids=chunk_generator.cache_chunk_ids(), cache_part_count=chunk_generator.cache_parts_cnt,
            replicas_chunk_ids=chunk_generator.replicas_chunk_ids(), replicas_part_count=chunk_generator.replicas_parts_cnt,
            timeouts_multiplicator=timeouts_multiplicator * _HAMSTER_TIMEOUTS_MULTIPLICATOR,
        )
        configs.update(
            generate_basesearch_configs(basesearch_hamster_agent_shard_number_map, endpoint_set_hamster_template, hamster_shard_configs)
        )

    configs.update(generate_remote_storage_configs(
        tier, remote_storage_host_agent_map, chunk_hosts_mapping, namespace_prefix, subresources
    ))

    return configs
