import collections
import datetime
import logging

import oldprod
import sandbox_tasks

import infra.callisto.controllers.sdk as sdk
from infra.callisto.controllers.sdk import tier, notify
from infra.callisto.controllers.build import innerbuilder
import infra.callisto.libraries.yt as yt_utils
import infra.callisto.controllers.sdk.registry as registry
import infra.callisto.controllers.slots.extctrl as extctrl
import infra.callisto.controllers.utils.gencfg_api as gencfg_api

import yt_observers
import pip
import vla
import vla_yp
import sas
import sas_platinum
import man
import man_yp
import yt_copier
import test

READY_FOR_PERF_LEVEL = 0.995


class JupiterController(sdk.Controller):
    builders_topology = 'stable-157-r3537'

    @property
    def path(self):
        return 'jupiter'

    def __init__(self, readonly):
        super(JupiterController, self).__init__()
        self.readonly = readonly

        self.builders = {
            tier.PlatinumTier0: innerbuilder.make_controller(
                yt_observers.PlatinumTier,
                instance_provider=gencfg_api.InstanceProvider(
                    [gencfg_api.GencfgGroup('VLA_WEB_PLATINUM_BUILD', self.builders_topology, mtn=True)],
                    report_tags={'VLA_WEB_PLATINUM_BUILD'}
                ),
                assist_last_n_on_finish=5,
                remove_failed_tasks=False,
            ),
            tier.WebTier0: innerbuilder.make_controller(
                yt_observers.WebTier0,
                instance_provider=gencfg_api.InstanceProvider(
                    [gencfg_api.GencfgGroup('VLA_WEB_TIER0_BUILD', self.builders_topology, mtn=True)],
                    report_tags={'VLA_WEB_TIER0_BUILD'}
                ),
                space_needed_full_build_coeff=1.2,
                assist_last_n_on_finish=72,
                max_builders_per_task=2,
                remove_failed_tasks=False,
            ),
            tier.WebTier0Mini: innerbuilder.make_controller(
                yt_observers.WebTier0Mini,
                instance_provider=gencfg_api.InstanceProvider(
                    [gencfg_api.GencfgGroup('VLA_WEB_TIER0_MINI_BUILD', self.builders_topology, mtn=True)],
                    report_tags={'VLA_WEB_TIER0_MINI_BUILD'}
                ),
                space_needed_full_build_coeff=1.2,
                assist_last_n_on_finish=72,
                max_builders_per_task=2,
                remove_failed_tasks=False,
            ),
            tier.MsUserData: innerbuilder.make_controller(
                yt_observers.MsUserDataTier,
                instance_provider=gencfg_api.InstanceProvider(
                    [gencfg_api.GencfgGroup('VLA_WEB_MMETA_BUILD', self.builders_topology, mtn=True)],
                    report_tags={'VLA_WEB_MMETA_BUILD'}
                ),
                assist_last_n_on_finish=1,
                max_builders_per_task=2,
                remove_failed_tasks=False,
            ),
            tier.JudTier: innerbuilder.make_controller(
                yt_observers.JudTier,
                instance_provider=gencfg_api.InstanceProvider(
                    [gencfg_api.GencfgGroup('VLA_WEB_JUD_BUILD', self.builders_topology, mtn=True)],
                    report_tags={'VLA_WEB_JUD_BUILD'}
                ),
                assist_last_n_on_finish=1,
                max_builders_per_task=2,
                remove_failed_tasks=False,
            ),
            tier.GeminiTier: innerbuilder.make_controller(
                yt_observers.Gemini,
                instance_provider=gencfg_api.InstanceProvider(
                    [gencfg_api.GencfgGroup('VLA_GEMINI_BUILD', self.builders_topology, mtn=True)],
                    report_tags={'VLA_GEMINI_BUILD'}
                ),
                space_needed_full_build_coeff=1,
                assist_last_n_on_finish=1,
                max_builders_per_task=2,
                remove_failed_tasks=False,
            ),
            tier.CastorTier: innerbuilder.make_controller(
                yt_observers.Castor,
                instance_provider=gencfg_api.InstanceProvider(
                    [gencfg_api.GencfgGroup('VLA_CASTOR_BUILD', self.builders_topology, mtn=True)],
                    report_tags={'VLA_CASTOR_BUILD'}
                ),
                assist_last_n_on_finish=1,
                max_builders_per_task=2,
                remove_failed_tasks=False,
            )
        }

        self.prod = oldprod.Controller()
        _ctrls = 'http://ctrl.clusterstate.yandex-team.ru'
        self.pip = extctrl.ExternalSlotsController(_ctrls + '/web/pip', readonly, pip.get_yt_target_table(readonly),
                                                   pip.get_yt_status_table(readonly=True))
        # self.man_web = extctrl.ExternalSlotsController(_ctrls + '/web/prod-man', readonly,
        #                                                man.get_yt_target_table(readonly))
        # self.man_yp = extctrl.ExternalSlotsController(_ctrls + '/web/prod-man-yp', readonly,
        #                                                man_yp.get_yt_target_table(readonly))
        self.sas_web = extctrl.ExternalSlotsController(_ctrls + '/web/prod-sas', readonly,
                                                       sas.get_yt_target_table(readonly))
        self.sas_platinum = extctrl.ExternalSlotsController(_ctrls + '/web/prod-sas-platinum', readonly,
                                                            sas_platinum.get_yt_target_table(readonly))
        self.vla_web = extctrl.ExternalSlotsController(_ctrls + '/web/prod-vla', readonly,
                                                       vla.get_yt_target_table(readonly))
        self.vla_yp = extctrl.ExternalSlotsController(_ctrls + '/web/prod-vla-yp', readonly,
                                                       vla_yp.get_yt_target_table(readonly))
        # self.man_test = extctrl.ExternalSlotsController(_ctrls + '/web/test-man', readonly,
        #                                                 test.target_table(readonly))

        yt_client = yt_utils.create_yt_client('arnold')
        self.yt_copiers = {
            tier.WebTier0: yt_copier.JupiterChunksCopierController(
                yt_observers.WebTier0,
                'web/prod/yt',
                yt_client,
                readonly
            ),
            tier.EmbeddingWebTier0: yt_copier.JupiterPlainCopierController(
                yt_observers.EmbeddingWebTier0,
                'web/prod/yt',
                yt_client,
                readonly,
                torrents_table_name='embedding_storage_torrents'
            ),
            tier.InvertedIndexWebTier0: yt_copier.JupiterPlainCopierController(
                yt_observers.InvertedIndexWebTier0,
                'web/prod/yt',
                yt_client,
                readonly,
                torrents_table_name='partitioned_panther_torrents'
            ),
            tier.KeyInvWebTier0: yt_copier.JupiterChunksCopierController(
                yt_observers.KeyInvWebTier0,
                'web/prod/yt',
                yt_client,
                readonly
            ),
            "embedding beta1": yt_copier.JupiterPlainCopierController(
                yt_observers.EmbeddingWebTier0,
                'web/prod/yt/beta1',
                yt_client,
                readonly,
                torrents_table_name='embedding_storage_torrents_1'
            ),
            "embedding beta2": yt_copier.JupiterPlainCopierController(
                yt_observers.EmbeddingWebTier0,
                'web/prod/yt/beta2',
                yt_client,
                readonly,
                torrents_table_name='embedding_storage_torrents_2'
            ),
            "embedding beta3": yt_copier.JupiterPlainCopierController(
                yt_observers.EmbeddingWebTier0,
                'web/prod/yt/beta3',
                yt_client,
                readonly,
                torrents_table_name='embedding_storage_torrents_3'
            ),
            "embedding beta4": yt_copier.JupiterPlainCopierController(
                yt_observers.EmbeddingWebTier0,
                'web/prod/yt/beta4',
                yt_client,
                readonly,
                torrents_table_name='embedding_storage_torrents_4'
            ),
            "embedding beta5": yt_copier.JupiterPlainCopierController(
                yt_observers.EmbeddingWebTier0,
                'web/prod/yt/beta5',
                yt_client,
                readonly,
                torrents_table_name='embedding_storage_torrents_5'
            ),
            "chunk-ctl-supplier": yt_copier.ChunkCtlSpecsController(
                yt_observers.WebTier0,
                yt_client,
                readonly
            )
        }

        self.register(
            self.prod,
            self.pip,
            # self.man_web,
            # self.man_yp,
            self.sas_web,
            self.sas_platinum,
            self.vla_web,
            self.vla_yp,
            # self.man_test,
            *(self.yt_copiers.values() + self.builders.values())
        )
        self.pip_deploy_ruler = make_deploy_ruler(max_states=1)
        self.prod_deploy_ruler = make_deploy_ruler(max_states=2)
        self.switcher = make_switcher(self.readonly)
        self._notifications = notify.NotificationsAggregator()
        self._unistat = {}

    def execute(self):
        unistat_signals = {}

        if self.prod.common_state:
            prod_timestamp = int(self.prod.common_state)
            builders_timestamps = self._common_builders_timestamps((tier.PlatinumTier0, tier.MsUserData))

            if 14 <= datetime.datetime.now().hour <= 15:
                self._set_prod(prod_timestamp)

            self._notifications.add_notifications(self.switcher.check_and_switch(prod_timestamp, unistat_signals))

            self._run_performance_test_task(builders_timestamps, prod_timestamp)
        else:
            logging.debug('No common state observed, leaving pip & prod intact')

        self._set_pip(int(self.prod.newest_state))

        self._unistat = unistat_signals
        self._notifications.push_buffer()

    def _common_builders_timestamps(self, tiers):
        builders = [builder for builder in self.builders.values() if builder.tier in tiers]
        timestamps = set(builders[0].timestamps())
        for builder in builders:
            timestamps &= set(builder.timestamps())
        return timestamps

    def _set_pip(self, prod_timestamp):
        deploy_target = self.pip_deploy_ruler.to_deploy_on_acceptance(prod_timestamp)
        basesearch_target = self.pip_deploy_ruler.search_target_on_acceptance(deploy_target, prod_timestamp)
        self.pip.set_target_state(deploy_target, basesearch_target, null_if_not_deployed=True)

    def _run_performance_test_task(self, build_timestamps, prod_timestamp):
        mmeta_builder = self.builders.get(tier.MsUserData)

        for check_tier in (tier.PlatinumTier0,):
            builder = self.builders[check_tier]

            for timestamp in build_timestamps:
                if timestamp == prod_timestamp:
                    continue  # Do not compare db with itself

                if not _tier_is_ready(mmeta_builder, timestamp):
                    logging.debug('Tier %s ts %s not ready, skip', tier.MsUserData.name, timestamp)
                    continue

                if not _tier_is_ready(builder, timestamp):
                    logging.debug('Tier %s ts %s not ready, skip', check_tier.name, timestamp)
                    continue

                if sandbox_tasks.is_performance_test_created(
                        timestamp,
                        prod_timestamp,
                        check_tier.name
                ):
                    logging.debug('Check for %s, ts %s found, skip', check_tier.name, timestamp)
                    continue

                if not self.readonly:
                    sandbox_tasks.create_performance_test(
                        timestamp,
                        prod_timestamp,
                        check_tier.name
                    )
                    logging.debug('Run check for %s, ts %s', check_tier.name, timestamp)
                else:
                    logging.debug(
                        'Need to run check for %s, ts %s, but read only mode',
                        check_tier.name, timestamp,
                    )

    def _set_prod(self, prod_timestamp):
        deploy_target = self.prod_deploy_ruler.to_deploy_on_prod(prod_timestamp)
        assert prod_timestamp in deploy_target

        if datetime.datetime.now().minute < 30:
            self.sas_web.set_target_state(deploy_target, prod_timestamp)
            self.sas_platinum.set_target_state(deploy_target, prod_timestamp)
        else:
            self.vla_web.set_target_state(deploy_target, prod_timestamp)
            self.vla_yp.set_target_state(deploy_target, prod_timestamp)

    def json_view(self):
        result = collections.OrderedDict()
        result['__readonly__'] = self.readonly
        # result['man_web'] = self.man_web.json_view()
        result['sas_web'] = self.sas_web.json_view()
        result['vla_web'] = self.vla_web.json_view()
        result['new_pip'] = self.pip.json_view()
        result['builders'] = {tier_.name: builder.json_view() for tier_, builder in self.builders.items()}
        result['prod'] = self.prod.json_view()
        return result

    def build_progress(self, tier_name):
        for _tier in self.builders:
            if _tier.name == tier_name:
                return self.builders[_tier].build_progress(tier_name)

    def notifications(self):
        return self._notifications.get_notifications()

    def unistat(self):
        return [
            [name, value]
            for name, value in self._unistat.items()
        ]


def _tier_is_ready(builder, timestamp):
    done_shards_count = builder.generations_status().get(timestamp, {}).get('DONE', 0)
    return done_shards_count > READY_FOR_PERF_LEVEL * builder.yt_observer.tier.shards_count


def get_acceptance_table():
    return sdk.prod_ruler.AcceptanceTableSource(
        path='//home/jupiter/shard_deploy/acceptance_prod',
        yt_client=yt_utils.create_yt_client('arnold')
    )


def make_deploy_ruler(max_states):
    return sdk.prod_ruler.DeployRuler(
        build_source=yt_observers.ProductionBaseSource,
        acceptance_source=get_acceptance_table(),
        max_states_to_keep=max_states,
    )


def make_search_ruler():
    return sdk.prod_ruler.SearchRuler(
        acceptance_source=get_acceptance_table(),
        target_tables=[
            location.get_yt_target_table(readonly=True)
            for location in (sas, vla)
        ],
        status_tables=[
            location.get_yt_status_table(readonly=True)
            for location in (sas, vla)
        ],
    )


def make_switcher(readonly):
    return sdk.prod_ruler.Switcher(
        search_ruler=make_search_ruler(),
        task_type='SWITCH_WEB_DB_INDEX',
        vertical_name='web',
        deadline_time=datetime.time(hour=1, minute=45),
        enable_autorelease=True,
        readonly=readonly,
    )


def make_ctrl(readonly):
    return JupiterController(readonly)


registry.register('web/prod/main', make_ctrl, [registry.ReportsBackends.V2.vla])
