import gevent.pool
import logging

import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.sdk.registry as registry
import infra.callisto.controllers.utils.gencfg_api as gencfg_api
import infra.callisto.libraries.yt as yt_utils

default_topology = 'stable-151-r1500'
memory_guarantee = 512 * 1024 * 1024
memory_limit = 1024 * 1024 * 1024

mmeta_ports_required = 3


class MetaCtrl(sdk.Controller):
    path = 'meta'

    def __init__(self, slaves, readonly=True):
        super(MetaCtrl, self).__init__()
        self.readonly = readonly
        self.register(*slaves)


class Ctrl(sdk.Controller):
    def __init__(self, location, topology, mmeta, idle):
        super(Ctrl, self).__init__()

        self.location = location.lower()
        self.topology = topology
        self.mmeta = mmeta
        self.idle = idle

    @property
    def path(self):
        return 'hamster@{}'.format(self.location)

    @property
    def is_ready(self):
        return not self._prod_is_locked()

    def _prod_is_locked(self):
        if _lock_exists('//home/cajuper/hamster/locks/{}'.format(self.location)):
            _log.debug('Location %s is LOCKED for db switch', self.location)
            return True

        return False

    def gencfg(self):
        configs = {}
        configs.update(_gencfg_mmeta(self.mmeta, self.is_ready))

        return configs

    def json_view(self):
        return {
            'topology': self.topology,
            'configs_sample': self.gencfg().items()[:5],
        }


def _lock_exists(lock):
    yt_client = yt_utils.create_yt_client('locke')
    return yt_client.exists(lock)


def _get_porto_property(config, prop):
    return config['porto_limits'][prop]


# currently using experimental idle configurations
# don't delete this config
def _get_container_description(config):
    return {
        'cpu_policy': 'normal',
        'recharge_on_pgfault': False,
        'cpu_limit': str(_get_porto_property(config, 'cpu_limit')),
        'cpu_guarantee': '2.0',  # str(_get_porto_property(config, 'cpu_guarantee')),
        'memory_limit': str(memory_limit),
        'memory_guarantee': str(memory_guarantee),
    }


def _limitless_idle_hamster():
    return {
        'cpu_policy': 'idle',
        'recharge_on_pgfault': False,
        'io_limit': '/ssd r: 104857600',
        'memory_limit': str(memory_limit),
        'memory_guarantee': str(memory_guarantee),
    }


def _limited_idle_prs_ops():
    return {
        'cpu_policy': 'idle',
        'cpu_limit': '2c',
        'io_limit': '/ssd r: 104857600',
        'recharge_on_pgfault': False,
        'memory_limit': str(memory_limit),
        'memory_guarantee': str(memory_guarantee),
    }


def _gencfg_mmeta(group, ready=True):
    return {
        (agent.host, agent.port): {
            'mapping': {
                'slots': {
                    'mmeta': {
                        'port': agent.port + mmeta_ports_required,
                        'enable': True,
                        'container': {},
                    },
                },
                'default': ('mmeta' if ready else ''),
            },
        } for agent, config in group.iteritems()
    }


def make_ctrl(location, idle=False):
    def resolve(group_def):
        return gencfg_api.searcher_lookup_agents(group_def['group'], default_topology, group_def['mtn'])

    (
        mmeta,
        mmeta_prs,
    ) = gevent.pool.Pool().map(resolve, [
        {'group': '{}_WEB_MMETA_HAMSTER'.format(location), 'mtn': True},
        {'group': '{}_WEB_MMETA_PRS_HAMSTER'.format(location), 'mtn': True},
    ])

    mmeta.update(mmeta_prs)

    return Ctrl(location, default_topology, mmeta, idle)


def make_meta_ctrl(readonly):
    return MetaCtrl(
        slaves=[make_ctrl(location, idle=True) for location in ('MAN', 'SAS', 'VLA')],
        readonly=readonly
    )


registry.register('web/prod/hamster', make_meta_ctrl, registry.ReportsBackends.V2.all, sleep_time=60)


_log = logging.getLogger('hamster')
