import infra.callisto.controllers.deployer2.controller as deploy_controller
import infra.callisto.controllers.deployer2.proxy as proxy
import infra.callisto.controllers.build.innerbuilder as innerbuilder
import infra.callisto.controllers.sdk.registry as registry
import infra.callisto.controllers.sdk as sdk

import infra.callisto.controllers.utils.entities as entities
import infra.callisto.controllers.utils.gencfg_api as gencfg_api
import infra.callisto.controllers.slots.state as slot_state
import infra.callisto.controllers.sdk.tier as tiers
import infra.callisto.libraries.yt as yt_utils
import yt_copier

from .man import get_yt_target_table
import chunks.controller
import chunks.allocator
import yt_observers
import tier1


def make_tier1_chunks_ctrl(
    deploy_ctrl,
    namespace_prefix,
    remote_storage_group,
    readonly,
    endpoint_set_template=None,
    endpoint_set_hamster_template=None,
):
    def _host_guarantee(instance):
        return int(instance.get('storages', {}).get('rootfs', {}).get('size', 0)) * 1024 ** 3
    chunks.controller._host_guarantee = _host_guarantee

    remote_storage_host_agent_map = gencfg_api.strict_host_agent_mapping([remote_storage_group], mtn=True)
    allocator = chunks.allocator.Allocator(
        location='erasure_test',
        tier=sdk.tier.get_erasure_test_tier(),
        readonly=readonly,
        max_shard_parts_on_host=2000,
        rack_antiaffinity=False
    )
    remote_storage_host_agent_map = gencfg_api.strict_host_agent_mapping([remote_storage_group], mtn=True)
    remote_storage_instances = gencfg_api.searcher_lookup_instances(*remote_storage_group)
    return chunks.controller.SlotCtrlAdapter(
        name='Tier1Chunks',
        deploy_ctrl=deploy_ctrl,
        namespace_prefix=namespace_prefix,
        allocator=allocator,
        chunks_generator=tier1.ErasureChunksGenerator(tier=tiers.get_erasure_test_tier()),
        remote_storage_host_agent_map=remote_storage_host_agent_map,
        basesearch_hamster_agent_shard_number_mappings={},
        basesearch_agent_shard_number_mappings={
            entities.Agent('vla2-9091.search.yandex.net', 29985): (0, 0),
            entities.Agent('vla2-9098.search.yandex.net', 29985): (0, 0),
        },
        remote_storage_instances=remote_storage_instances,
        generation_max_share=1,
        chunk_resources=('arc', 'extarc', 'keyinv'),
        endpoint_set_template=endpoint_set_template,
        endpoint_set_hamster_template=endpoint_set_hamster_template,
    )


class DeployProxy(proxy.ChunksProxy):
    timestamps = set()

    def host_chunks_target(self):
        shards = [
            sdk.tier.get_erasure_test_tier().make_shard((0, 0), timestamp)
            for timestamp in self.timestamps
        ]
        return {
            host: [
                entities.Chunk(shard, 'local')
                for shard in shards
            ]
            for host in [
                'sas1-6720.search.yandex.net',
                'vla2-9098.search.yandex.net'
            ]
        }


class Ctrl(sdk.Controller):
    path = 'DevDeploy'

    def __init__(self, readonly):
        super(Ctrl, self).__init__()
        namespace_prefix = '/web/betas/erasure/'
        topology = 'stable-130-r589'
        self._deployer = deploy_controller.DeployerController(
            gencfg_api.get_strict_host_agent_mapping([
                ('SAS_SSMIKE_DEV', topology),
                ('VLA_ERASURE_BETA_DEPLOY', topology),
                ('DEV_DEPLOY', 'stable-133-r238')
            ]),
            download_args_callback=lambda shard, host: {'max_dl_speed': '120M'},
            tags={'a_tier_none'},
        )
        self._proxy = DeployProxy(self._deployer, namespace_prefix, ('',))
        self._proxy.chunk_resources = ('',)
        self._remote_storage_adapter = make_tier1_chunks_ctrl(
            self._deployer,
            'web/betas/erasure/remote_storage',
            ('VLA_SEARCH_ERASURE_BETA', topology),
            readonly,
            # endpoint_set_template='man@man-web-search.man-web-tier1-base.{}',
            # endpoint_set_hamster_template='man@man-web-search.man-web-tier1-base.{}.hamster',
        )
        self._target_table = get_yt_target_table(readonly=True)
        self._builder = innerbuilder.make_controller(
            yt_observers.get_erasure_test_webtier1(),
            instance_provider=gencfg_api.InstanceProvider(
                [gencfg_api.GencfgGroup('VLA_SEARCH_ERASURE_BUILDER_BETA', 'stable-137-r120', mtn=True)],
                report_tags={'VLA_SEARCH_ERASURE_BUILDER_BETA'}
            ),
            assist_last_n_on_finish=1,
            remove_failed_tasks=False,
            ignore_guarantees=True
        )

        yt_client = yt_utils.create_yt_client('arnold')
        self.yt_copiers = {
            sdk.tier.WebTier1: yt_copier.JupiterChunksCopierController(
                yt_observers.WebTier1,
                'web/betas/erasure/remote_storage',
                yt_client,
                readonly
            ),
        }

        self.register(self._deployer, self._proxy, self._remote_storage_adapter, self._builder, *self.yt_copiers.values())

    def execute(self):
        self._proxy.timestamps = self._target_table.head().target.deploy
        self._remote_storage_adapter.set_target_state(
            [slot_state.SlotState(ts) for ts in self._proxy.timestamps],
            None
        )


def make_controller(readonly):
    return Ctrl(readonly)


registry.register('web/prod/new_base', make_controller, [registry.ReportsBackends.V2.sas])
