import collections
import logging

import infra.nanny.nanny_services_rest.nanny_services_rest.client as nanny

import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.utils.secrets as secrets
import infra.callisto.controllers.utils.sandbox_utils as sandbox_utils


class Controller(sdk.Controller):
    path = 'oldprod'

    def __init__(self):
        super(Controller, self).__init__()
        self._locations = {
            # 'man': LocationObserver('man-web-base-resources', 'man_jupiter_mmeta_yp', 'man-web-int-resources'),
            'sas': LocationObserver('sas-web-base-resources', 'sas_jupiter_mmeta_yp', 'sas-web-int-resources'),
            'vla': LocationObserver('vla-web-base-resources', 'vla_jupiter_mmeta_yp', 'vla-web-int-resources'),
        }
        self._last_known_common_config = None

    def update(self, reports):
        for location in self._locations.itervalues():
            location.update()

    def json_view(self):
        json = {
            location: _state_to_json(observer.state)
            for location, observer in self._locations.iteritems()
        }
        json['common'] = _state_to_json(LocationState(self.common_state, self.common_config))
        return json

    @property
    def common_state(self):
        state = self._locations['sas'].state.yt_state
        for observer in self._locations.itervalues():
            if state != observer.state.yt_state:
                logging.debug('Common state is None')
                return None

        logging.debug('Common state: %s', state)
        return state

    @property
    def newest_state(self):
        return max(observer.state.yt_state for observer in self._locations.itervalues())

    @property
    def common_config(self):
        config = self._locations['sas'].state.config
        for observer in self._locations.itervalues():
            if config != observer.state.config:
                logging.debug('There is no common config, return cached %s', self._last_known_common_config)
                return self._last_known_common_config

        logging.debug('Common config: %s', config)
        self._last_known_common_config = config

        return config

    def __str__(self):
        return 'OldProdController'


class LocationObserver(object):
    def __init__(self, base_service, mmeta_service, int_service):
        self.base_service = base_service
        self.int_service = int_service
        self.mmeta_service = mmeta_service
        self.state = None
        self._state_resource = None
        self._yt_state = None

    def update(self):
        self._update_yt_state()

        self.state = LocationState(
            yt_state=self._yt_state,
            config=LocationConfig(
                base=_get_httpsearch(self.base_service),
                mmeta=_get_httpsearch(self.mmeta_service),
                int=_get_intsearch(self.int_service),
            )
        )

    def _update_yt_state(self):
        try:
            self._state_resource, self._yt_state = self._get_yt_state(self.base_service)
        except RuntimeError:
            logging.exception('Could not update yt state for %s', self.base_service)

    def _get_yt_state(self, base_service):
        current_state_resource = _get_vars_conf(base_service)
        if current_state_resource and self._state_resource != current_state_resource:
            logging.info(
                'Found new state resource[id=%s] at service %s',
                current_state_resource['resource_id'],
                base_service
            )
            task_details = sandbox_utils.get_task(current_state_resource['task_id'])
            current_db_timestamp = task_details.get('input_parameters', {}).get('db_timestamp')
            if current_db_timestamp:
                return current_state_resource, str(current_db_timestamp)
            else:
                logging.debug('Could not get task[id=%s] details', current_state_resource['task_id'])

            raise RuntimeError('Could not get yt state for %s', base_service)
        else:
            return self._state_resource, self._yt_state


LocationState = collections.namedtuple('LocationState', ['yt_state', 'config'])
LocationConfig = collections.namedtuple('LocationConfig', ['base', 'mmeta', 'int'])


def _state_to_json(state):
    return {
        'yt_state': state.yt_state,
        'config': {
            'base': state.config.base,
            'mmeta': state.config.mmeta,
        } if state.config else None
    }


def _get_httpsearch(service):
    resources = (
        'httpsearch',
        'models.archive',  # gdb? evlogdump?
    )

    return _get_nanny_resources(service, resources)


def _get_intsearch(service):
    resources = (
        'httpsearch',
    )

    return _get_nanny_resources(service, resources)


def _get_nanny_resources(service, resource_local_names):
    def filter_fields(nanny_dict):
        return {key: nanny_dict.get(key) for key in {'resource_id', 'task_id'}}

    result = dict.fromkeys(resource_local_names)

    for sandbox_file in _get_service_sandbox_files(service):
        if sandbox_file['local_path'] in result:
            result[sandbox_file['local_path']] = filter_fields(sandbox_file)

    return result


def _get_vars_conf(service):
    for sandbox_resource in _get_service_sandbox_files(service):
        if sandbox_resource['resource_type'] == 'WEB_PRODUCTION_INDEX_STATE_RESOURCE':
            return sandbox_resource


def _get_service_sandbox_files(service):
    return _get_service_resources(service)['sandbox_files']


def _get_service_resources(service):
    nanny_client = nanny.ServiceRepoClient(
        NANNY_URL,
        secrets.find_nanny_token(),
        attempts=5,
        timeout=1,
        delay=1,
    )
    return nanny_client.get_active_runtime_attrs(service)['content']['resources']


NANNY_URL = 'https://nanny.yandex-team.ru'
