import infra.callisto.libraries.memoize as memoize
import infra.callisto.controllers.utils.sandbox_utils as sandbox_utils


PERFORMANCE_TASK_TYPE = 'PRIEMKA_BASESEARCH_DATABASE_JUPITER'


@memoize.memoized
def is_performance_test_created(db_timestamp, prod_timestamp, tier):
    sandbox = sandbox_utils.get_sandbox_client()
    tasks_found = sandbox.task.read(
        type=PERFORMANCE_TASK_TYPE,
        limit=1,
        tags=[
            tier,
            'prod_{}'.format(prod_timestamp),
            'newdb_{}'.format(db_timestamp)
        ],
        all_tags=True,
    )
    if tasks_found.get('total') >= 1:
        return True
    return memoize.DontSave(False)


def create_performance_test(db_timestamp, prod_timestamp, tier):
    RESOURCES = {
        'BASESEARCH_CONFIG': None,
        'BASESEARCH_EXECUTABLE': None,
        'ISS_SHARDS': 113291061,
        'MIDDLESEARCH_EXECUTABLE': None,
    }
    REQUEST_LIMIT = 100000
    SHARDS_PER_TIER = 30
    PERFORMANCE_TEST_DESCRIPTION = 'Performance test for {0}, timestamp: {1} vs prod: {2}'

    sandbox = sandbox_utils.get_sandbox_client()
    task_id = sandbox.task({'type': PERFORMANCE_TASK_TYPE})['id']

    sandbox.task[task_id] = {
        'owner': 'SEPE-PRIEMKA',
        'description': PERFORMANCE_TEST_DESCRIPTION.format(
            tier,
            db_timestamp,
            prod_timestamp
        ),
        'priority': {
            'subclass': 'HIGH',
            'class': 'SERVICE'
        },
        'requirements': {
            u'disk_space': 62914560000,
        },
        'custom_fields': [
            {'name': 'run_tests_with_factors_plan', 'value': True},
            {'name': 'measure_middlesearch_memory_usage', 'value': False},
            {'name': 'executor_requests_limit', 'value': REQUEST_LIMIT},
            {'name': 'tier_shard_count', 'value': SHARDS_PER_TIER},
            {'name': 'test_tier_PlatinumTier0', 'value': (tier == 'PlatinumTier0')},
            {'name': 'test_tier_WebTier0', 'value': False},
            {'name': 'test_tier_WebTier1', 'value': (tier == 'WebTier1')},
            {'name': 'basesearch_executable_resource_id', 'value': RESOURCES['BASESEARCH_EXECUTABLE']},
            {'name': 'basesearch_config_resource_id', 'value': RESOURCES['BASESEARCH_CONFIG']},
            {'name': 'middlesearch_executable_resource_id', 'value': RESOURCES['MIDDLESEARCH_EXECUTABLE']},
            {'name': 'iss_shards', 'value': RESOURCES['ISS_SHARDS']},
            {'name': 'override_newdb_timestamp', 'value': db_timestamp}
        ],
        'tags': [
            tier,
            'prod_{}'.format(prod_timestamp),
            'newdb_{}'.format(db_timestamp)
        ],
    }
    sandbox.batch.tasks.start.update(task_id)

    return task_id


def generate_rs_configs_task(location, db_timestamp, topology, tier, namespace_prefix, subresources,
                             rs_group, basesearch_groups, basesearch_hamsters_groups=None,
                             endpoint_set_template=None, endpoint_set_hamster_template=None,
                             readonly=True):
    task_type = 'GENERATE_CAJUPER_CHUNKS_CONFIGS'

    return sandbox_utils.run_only_task(
        task_type,
        {
            'location': location,
            'db_timestamp': db_timestamp,
            'topology': topology,
            'namespace_prefix': namespace_prefix,
            'chunk_namespace_prefix': 'web/prod/chunks',  # TODO: fix hardcoded value
            'subresources': subresources,
            'remote_storage_group': rs_group,
            'basesearch_groups': basesearch_groups,
            'basesearch_hamster_groups': basesearch_hamsters_groups or [],
            'tier_name': tier.name,
            'endpoint_set_template': endpoint_set_template or '',
            'endpoint_set_hamster_template': endpoint_set_hamster_template or '',
            'timeouts_multiplicator': 10.0 if location == 'pip' else 1.0,
        },
        'Generate parts configs for {}, db timestamp {}'.format(location, db_timestamp),
        tags=['web', location, tier.name],
        hints=[db_timestamp],
        readonly=readonly,
    )


def switch_pip_configs_task(db_timestamp, readonly):
    task_type = 'SWITCH_PIP_CONFIGS'

    return sandbox_utils.run_only_task(
        task_type,
        {'db_timestamp': int(db_timestamp)},
        'Switch remote storage configs to {} at PIP'.format(db_timestamp),
        auto_release=True,
        tags=['web', 'pip'],
        hints=['newdb_{}'.format(db_timestamp)],
        readonly=readonly
    )
