import collections
import datetime
import logging

import infra.callisto.controllers.deployer2.light as deployer_light
import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.sdk.tier as tiers
import infra.callisto.controllers.sdk.registry as registry
import infra.callisto.controllers.slots as slots
import infra.callisto.controllers.utils.entities as entities
import infra.callisto.controllers.utils.yp_utils as yp_utils

import tables


def make_test_stage_controller(readonly):
    stage = 'web-search-test'
    deploy_ctrl = deployer_light.make_controller(
        _get_deployer_agents('{}.platinum-base'.format(stage), hermetic=True),
        download_params={'max_dl_speed': '1000M',
                         'check_policy': {'repeat_sky_get': False},
                         'copier_opts': '{direct_write: 1, subproc: 1, max_write_chunk: 131072}',
                         'hardlink': True},
        tags=['a_itype_deployer', stage],
        reports_alive_threshold=datetime.timedelta(hours=2)
    )

    platinum = _YpBaseSlot(deploy_ctrl, 'man-pre', stage, '{}.platinum-base'.format(stage),
                           tiers.PlatinumTier0, hermetic=True, readonly=readonly)
    deploy_ctrl.register(platinum)

    return deploy_ctrl


class _YpBaseSlot(sdk.Controller):
    path = 'man_pre'

    def __init__(self, deploy_ctrl, cluster, stage_id, pod_set_id, tier, hermetic, readonly):
        super(_YpBaseSlot, self).__init__()
        self.cluster = cluster
        self.stage_id = stage_id
        self.pod_set_id = pod_set_id
        self.tier = tier
        self.hermetic = hermetic
        self.readonly = readonly

        if deploy_ctrl:
            deploy_ctrl.register_source(self._download_callback)

        self.log = logging.getLogger(__name__)  # use path/id

        # hardcoded for dev
        self.namespace_prefix = 'web/prod'

        # sources
        self.db_target_table = target_table(self.readonly)

        # runtime fields
        self._pods = []
        self._target = None

    def update(self, _):
        self._pods = self._read_pods()
        self._target = self.db_target_table.head().target

    def _download_callback(self):
        resources_on_node = collections.defaultdict(list)
        for pod in self._pods:
            for target_timestamp in self._target.deploy:
                shard = self.tier.make_shard(pod.pod_index % self.tier.shards_count, target_timestamp)
                resources_on_node[pod.node].append(self._shard_to_resource(shard))
        return resources_on_node.iteritems()

    # Workaround.
    def _shard_to_resource(self, shard):
        if self.tier == tiers.WebTier1:
            return slots.WebTier1Slot.shard_to_resource(self.namespace_prefix, shard)
        return sdk.resource.shard_to_resource(self.namespace_prefix, shard)

    def _read_pods(self):
        with self._yp_client() as client:
            records = client.select_objects('pod',
                                            filter='[/meta/pod_set_id]="{}"'.format(self.pod_set_id),
                                            selectors=['/meta/id', '/labels/pod_index',
                                                       '/status/dns/persistent_fqdn', '/spec/node_id'])
        return [PodFields(record[0], record[1], record[2], record[2] if self.hermetic else record[3])
                for record in records]

    def _yp_client(self):
        return yp_utils.client(yp_utils.YpMasters.sas) if self.cluster == 'sas' else yp_utils.man_pre()


PodFields = collections.namedtuple('PodFields', ['id', 'pod_index', 'fqdn', 'node'])


def _get_deployer_agents(pod_set_id, hermetic=False):
    with yp_utils.client(yp_utils.YpMasters.man_pre) as client:
        return [entities.Agent(host=obj[0],
                               node_name=obj[0] if hermetic else obj[1],
                               port=10000) for obj in
                client.select_objects('pod',
                                      filter='[/meta/pod_set_id]="{}"'.format(pod_set_id),
                                      selectors=['/status/dns/persistent_fqdn', '/spec/node_id'])]


test_stage_ctrl_id = 'web/test/man_pre'


def target_table(readonly):
    return tables.target(test_stage_ctrl_id, readonly)


def status_table(readonly):
    return tables.status(test_stage_ctrl_id, readonly)


registry.register(test_stage_ctrl_id, make_test_stage_controller, [registry.ReportsBackends.V2.man])
