import infra.callisto.controllers.sdk.tier as tiers
import infra.callisto.controllers.utils.entities as entities
import infra.callisto.controllers.utils.gencfg_api as gencfg_api
from itertools import chain

import chunks.controller
import chunks.allocator

CHUNK_NAMESPACE_PREFIX = '/web/prod/chunks'


class ErasureChunksGenerator(object):
    def __init__(self, tier=tiers.WebTier1, chunk_size_override=None, shard_filter=None):
        self.tier = tier
        self.size_override = chunk_size_override
        self.shard_filter = shard_filter

        self.erasure_chunks_cnt = 72
        self.erasure_parts_cnt = 16

        self.cache_chunks_cnt = 72
        self.cache_parts_cnt = 1

        self.replicas_chunks_cnt = 72
        self.replicas_parts_cnt = 1
        self.replicas_parts_replication = 3

        self.chunks_size = 0.92 * (1024 ** 3)
        self.replicas_chunks_size = 0.3 * (1024 ** 3)

        self._shard_count = 0
        for shard in self.tier.list_shards(0):
            if not self.skip_shard(shard):
                self._shard_count += 1

    def skip_shard(self, shard):
        return not (self.shard_filter is None) and shard.shard_number not in self.shard_filter

    def shards_count(self):
        return self._shard_count

    def list_erasure_chunks(self, generation):
        lst = []
        for shard in self.tier.list_shards(generation):
            if self.skip_shard(shard):
                continue
            for i in self.erasure_chunk_ids():
                for j in xrange(self.erasure_parts_cnt):
                    lst.append(entities.Chunk(
                        shard,
                        'remote_storage/{}/{}'.format(i, j),
                    ))
        return lst

    def list_replicas_chunks(self, generation):
        lst = []

        for shard in self.tier.list_shards(generation):
            if self.skip_shard(shard):
                continue
            for chunk_num in self.replicas_chunk_ids():
                for part_num in xrange(self.replicas_parts_cnt):
                    lst.append(entities.Chunk(
                        shard,
                        'multi_level_cache_replicas/{}/{}'.format(chunk_num, part_num),
                    ))

        return lst

    def chunk_size(self, chunk):
        if self.size_override is None:
            if chunk.number in self.replicas_chunk_ids():
                return self.replicas_chunks_size
            return self.chunks_size
        else:
            return self.size_override

    def erasure_chunk_ids(self):
        return xrange(0, self.erasure_chunks_cnt)

    def cache_chunk_ids(self):
        return xrange(self.erasure_chunks_cnt, self.erasure_chunks_cnt + self.cache_chunks_cnt)

    def replicas_chunk_ids(self):
        first_replicas_chunk_num = self.erasure_chunks_cnt + self.cache_chunks_cnt
        last_replicas_chunk_num = first_replicas_chunk_num + self.replicas_chunks_cnt
        return xrange(first_replicas_chunk_num, last_replicas_chunk_num)

    def chunk_parts_count(self, chunk):
        if chunk.number in self.erasure_chunk_ids():
            return self.erasure_parts_cnt
        elif chunk.number in self.cache_chunk_ids():
            return self.cache_parts_cnt
        elif chunk.number in self.replicas_chunk_ids():
            return self.replicas_parts_cnt
        raise ValueError('Unknown chunk with {} number'.format(chunk.number))

    def chunk_replication(self, chunk):
        if chunk.number in self.replicas_chunk_ids():
            return self.replicas_parts_replication
        return 1

    def total_size(self, generation, enable_replicas=True):
        iterable = chain(self.list_erasure_chunks(generation), [] if not enable_replicas else self.list_replicas_chunks(generation))
        return sum(self.chunk_size(i) for i in iterable)


class ErasureTier0ChunksGenerator(ErasureChunksGenerator):
    def __init__(self, chunk_size_override=None, shard_filter=None):
        super(ErasureTier0ChunksGenerator, self).__init__(tiers.WebTier0, chunk_size_override, shard_filter=shard_filter)
        self.erasure_chunks_cnt = 1476
        self.cache_chunks_cnt = 1476
        self.replicas_chunks_cnt = 1476
        self.chunks_size = 0.87 * (1024 ** 3)
        self.replicas_chunks_size = 0.91 * (1024 ** 3)


def make_chunk_generator(tier, chunk_size_override=None, shard_filter=None):
    if tier == tiers.WebTier0:
        return ErasureTier0ChunksGenerator(chunk_size_override=chunk_size_override, shard_filter=shard_filter)
    elif tier == tiers.WebTier1:
        return ErasureChunksGenerator(tier, chunk_size_override=chunk_size_override, shard_filter=shard_filter)
    RuntimeError('Tier {} is not supported'.format(tier))


def make_chunks_ctrl(
    location,
    deploy_ctrl,
    namespace_prefix,
    remote_storage_group,
    readonly=True,
    remote_storage_slots=None,
    generation_max_space_share=1.0,
    subresources=('',),
    chunk_size_override=None,
    configs_generator=None,
    tier=tiers.WebTier1,
    enable_replicas=True,
    is_optional=False,
    shard_filter=None,
    **allocator_args
):
    remote_storage_host_agent_map = gencfg_api.strict_host_agent_mapping([remote_storage_group], mtn=True)
    if remote_storage_slots is None:
        remote_storage_slots = [remote_storage_group]

    remote_storage_instances = []
    for group in remote_storage_slots:
        for instance in gencfg_api.searcher_lookup_instances(*group):
            remote_storage_instances.append(instance)

    return chunks.controller.SlotCtrlAdapter(
        name='{}Chunks'.format(tier.name),
        deploy_ctrl=deploy_ctrl,
        namespace_prefix=namespace_prefix,
        allocator=chunks.allocator.Allocator(location, tier, CHUNK_NAMESPACE_PREFIX, readonly, **allocator_args),
        chunks_generator=make_chunk_generator(tier, chunk_size_override=chunk_size_override, shard_filter=shard_filter),
        remote_storage_instances=remote_storage_instances,
        remote_storage_host_agent_map=remote_storage_host_agent_map,
        generation_max_share=generation_max_space_share,
        chunk_resources=subresources,
        configs_generator=configs_generator,
        enable_deploy_multi_level_cache_replicas=enable_replicas,
        is_optional=is_optional,
    )
