import cityhash
import datetime
import functools

import infra.callisto.controllers.deployer2.controller as deploy_controller
import infra.callisto.controllers.sdk.registry as registry
import infra.callisto.controllers.sdk.tier as tier
import infra.callisto.controllers.search_source.controller as search_source
import infra.callisto.controllers.slots as slots
import infra.callisto.controllers.utils.gencfg_api as gencfg_api
import infra.callisto.libraries.yt as yt_utils
import chunks.external as external_chunks

import deploy_params
import sandbox_tasks
import tier1


LOCATION = 'vla'
default_topology = 'stable-157-r5456'
deployer_groups = (
    ('VLA_WEB_DEPLOY', default_topology),
)

TIER0_BASESEARCH_GROUP = 'VLA_WEB_TIER0_BASE'
TIER0_ATTR_BASESEARCH_GROUP = 'VLA_WEB_TIER0_ATTRIBUTE_BASE'
TIER0_REMOTE_STORAGE_GROUP = 'VLA_WEB_TIER0_REMOTE_STORAGE_BASE'
TIER0_ENDPOINT_SET_TEMPLATE = 'vla@vla-web-search.tier0-base.{}'

TIER0_BASESEARCH_HAMSTER_GROUPS = [
    'VLA_WEB_TIER0_ATTRIBUTE_BASE_HAMSTER',
    'VLA_WEB_TIER0_BASE_HAMSTER',
    'VLA_WEB_TIER0_ATTRIBUTE_BASE_MULTI',
    'VLA_WEB_TIER0_BASE_MULTI',
]
TIER0_ENDPOINT_SET_HAMSTER_TEMPLATE = 'vla@vla-web-search.tier0-base.{}.hamster'

NAMESPACE_PREFIX = '/web/prod/'
NAMESPACE_PREFIX_YT = '/web/prod/yt/'
SUBRESOURCES = ('pudge', 'dynamic')


def slots_(topology=default_topology):
    return [
        slots.Slot('Platinum', tier.PlatinumTier0, 'VLA_WEB_PLATINUM_JUPITER_BASE', topology),
        slots.WebTier1Slot('WebTier0', tier.WebTier0, TIER0_BASESEARCH_GROUP, topology),
        slots.WebTier1Slot('WebTier0Attr', tier.AttributeWebTier0, TIER0_ATTR_BASESEARCH_GROUP, topology),
        slots.Slot('JudTier', tier.JudTier, 'VLA_WEB_JUD_JUPITER_BASE_NIDX', topology),
        slots.GeminiSlot('Gemini', tier.GeminiTier, 'VLA_WEB_GEMINI_BASE', topology),
        slots.Slot('CastorSearchProxy', tier.CastorTier, 'VLA_WEB_GEMINI_SEARCHPROXY_DYNA', topology),

        slots.Slot('Mmeta', tier.MsUserData, 'VLA_WEB_MMETA', topology),
        slots.Slot('HamsterPrsMmeta', tier.MsUserData, 'VLA_WEB_MMETA_PRS_HAMSTER', topology),

        slots.Slot('MmetaJprod', tier.MsUserData, 'VLA_WEB_MMETA_JPROD', topology),
        slots.Slot('MultiInt2Mmeta', tier.MsUserData, 'VLA_MULTIINT2_MMETA', topology),
        slots.Slot('MultiInt3Mmeta', tier.MsUserData, 'VLA_MULTIINT3_MMETA', topology),

        slots.Slot('CallistoBetaMmeta', tier.MsUserData, 'VLA_CALLISTO_MERCURY_MMETA', topology),
    ] + [
        slots.Slot('MultiBeta{}Mmeta'.format(num), tier.MsUserData, 'VLA_MULTIBETA{}_MMETA'.format(num), topology)
        for num in range(4, 8)
    ] + [
        slots.Slot('MultiMeta{}Mmeta'.format(num), tier.MsUserData, 'VLA_MULTIMETA{}_MMETA'.format(num), topology)
        for num in range(9, 45)
    ]


def yt_slots_(topology=default_topology):
    return [
        slots.WebTier1Slot(
            'KeyInvWebTier0', tier.KeyInvWebTier0,
            'VLA_WEB_TIER0_KEYINV', topology),
        slots.Slot('EmbeddingWebTier0', tier.EmbeddingWebTier0,
                   'VLA_WEB_TIER0_EMBEDDING', topology),
        slots.Slot('InvertedIndexWebTier0', tier.InvertedIndexWebTier0,
                   'VLA_WEB_TIER0_INVERTED_INDEX', topology),
    ]


def tier0_rs_slots(topology=default_topology):
    return (
        ('VLA_WEB_TIER0_REMOTE_STORAGE_SLOTS_0', topology),
        ('VLA_WEB_TIER0_REMOTE_STORAGE_SLOTS_1', topology),
        ('VLA_WEB_TIER0_REMOTE_STORAGE_SLOTS_2', topology),
    )


def embedding_beta_ctrls(topology, deploy_ctrl):
    beta_ctrls = {}
    for slot_index in range(1, 6):
        slot = slots.Slot('EmbeddingBetaWebTier0-{}'.format(slot_index),
                          tier.EmbeddingWebTier0,
                          'VLA_WEB_TIER0_EMBEDDING_BETA{}'.format(slot_index),
                          topology)
        namespace = NAMESPACE_PREFIX_YT + 'beta{}/'.format(slot_index)
        beta_ctrls.update(search_source.make_slot_controllers([slot], deploy_ctrl, deploy_only=True, namespace_prefix=namespace))
    return beta_ctrls


def get_yt_target_table(readonly):
    yt_client = yt_utils.create_yt_client('arnold', use_rpc=True)
    return search_source.get_yt_target_table(yt_client, 'web/prod/vla/target', readonly)


def get_yt_status_table(readonly):
    yt_client = yt_utils.create_yt_client('arnold', use_rpc=True)
    return search_source.get_yt_status_table(yt_client, 'web/prod/vla/status', readonly)


def make_controller(readonly):
    callback = deploy_params.DownloadArgs(gencfg_api.searcher_lookup_agents(*deployer_groups[0]))
    deploy_ctrl = deploy_controller.make_controller(deployer_groups, download_args_callback=callback, mtn=True)
    deploy_ctrl.reports_alive_threshold = datetime.timedelta(hours=1)
    slot_ctrls = search_source.make_slot_controllers(
        slots_(),
        deploy_ctrl,
        deploy_only=True,  # to not intersect with multibeta
        namespace_prefix=NAMESPACE_PREFIX,
    )
    slot_ctrls.update(search_source.make_slot_controllers(
        yt_slots_(),
        deploy_ctrl,
        deploy_only=True,  # to not intersect with multibeta
        namespace_prefix=NAMESPACE_PREFIX_YT))
    slot_ctrls.update(embedding_beta_ctrls(default_topology, deploy_ctrl))
    slot_ctrls['tier0_chunks'] = tier1.make_chunks_ctrl(
        location=LOCATION,
        deploy_ctrl=deploy_ctrl,
        namespace_prefix=NAMESPACE_PREFIX_YT,
        remote_storage_group=(TIER0_REMOTE_STORAGE_GROUP, default_topology),
        readonly=readonly,
        remote_storage_slots=tier0_rs_slots(default_topology),
        generation_max_space_share=0.5,
        subresources=SUBRESOURCES,
        configs_generator=functools.partial(
            sandbox_tasks.generate_rs_configs_task,
            location=LOCATION,
            topology=default_topology,
            tier=tier.WebTier0,
            namespace_prefix=NAMESPACE_PREFIX_YT,
            subresources=SUBRESOURCES,
            rs_group=TIER0_REMOTE_STORAGE_GROUP,
            basesearch_groups=[TIER0_BASESEARCH_GROUP, TIER0_ATTR_BASESEARCH_GROUP],
            basesearch_hamsters_groups=TIER0_BASESEARCH_HAMSTER_GROUPS,
            endpoint_set_template=TIER0_ENDPOINT_SET_TEMPLATE,
            endpoint_set_hamster_template=TIER0_ENDPOINT_SET_HAMSTER_TEMPLATE,
            readonly=readonly
        ),
        tier=tier.WebTier0,
        max_shard_parts_on_host=600,
        hash_function=cityhash.hash64,
        planner_version=2,
        is_optional=True,
    )
    slot_ctrls['tier0_chunks_ext'] = external_chunks.Controller(
        external_chunks.location_status('vla'),
        external_chunks.location_target('vla'),
        tier=tier.WebTier0,
        optional=False,
        readonly=readonly,
    )

    return search_source.YtDrivenSourceController(
        name='vla_web',
        slots_=slot_ctrls,
        deployer=deploy_ctrl,
        target_table=get_yt_target_table(readonly),
        status_table=get_yt_status_table(readonly),
    )


registry.register('web/prod/vla', make_controller, [registry.ReportsBackends.V2.vla])
