import datetime
import logging

import yt.wrapper as yt
from yt.yson import YsonUint64

from tier1 import ErasureChunksGenerator, ErasureTier0ChunksGenerator

from infra.callisto.deploy.tracker.core.table import NamespaceTable, namespace_to_path
from infra.callisto.controllers.utils.funcs import timestamp_to_yt_state
import infra.callisto.controllers.sdk as sdk

from search.plutonium.deploy.chunks.proto.chunks_pb2 import TChunkCtlResourceSpec
import search.base_search.common.protos.docs_tier_pb2 as docs_tier
import search.base.blob_storage.config.protos.chunk_access_type_pb2 as access_type

_log = logging.getLogger(__name__)

JUPITER_MRPREFIX = '//home/jupiter/shards_prepare'
SPECS_TABLE = '//home/cajuper/user/web/prod/chunks/ctl/specs'
SPEC_PARTS = '//home/cajuper/user/web/prod/chunks/ctl/spec_parts'


def discover_buckets(mr_prefix):
    for node in yt.list(mr_prefix):
        if node.isdigit():
            yield '/'.join((mr_prefix, node))


def torrent_tables_generator(mr_prefix, enable_merged=True):
    for bucket in discover_buckets(mr_prefix):
        yield bucket + '/dynamic_resources_torrents'
        yield bucket + '/pudge_resources_torrents'
        if enable_merged:
            yield bucket + '/merged_torrents'


def prepare_chunk_ctl_specs(timestamp, yt_client=yt):
    mr_prefix = '{}/{}'.format(JUPITER_MRPREFIX, timestamp_to_yt_state(timestamp))
    ensure_path(yt_client, SPEC_PARTS)
    spec_tmp_part = SPEC_PARTS + '/' + str(timestamp) + ".tmp"
    spec_merged_marker = SPEC_PARTS + '/' + str(timestamp) + ".merged"

    if yt_client.exists(spec_merged_marker):
        return

    def reducer(key, rows):
        shard, oth = key['Name'].split('/', 1)

        spec = TChunkCtlResourceSpec()
        spec.Labels.Label["AccessType"] = str(access_type.Disk)
        if 'Tier0' in shard:
            generator = ErasureTier0ChunksGenerator()
            spec.Labels.Label["Tier"] = str(docs_tier.WebTier0)
        else:
            generator = ErasureChunksGenerator()
            spec.Labels.Label["Tier"] = str(docs_tier.WebTier1)

        for row in rows:
            part = spec.Source.Compound.Sources.add()
            part.Source.Skynet.Url = row['Rbtorrent']
            part.Source.Skynet.Size = row['Size']

        spec.Labels.Label["Shard"] = shard.rsplit('-', 1)[1]
        spec.Labels.Label["Version"] = str(timestamp)
        spec.QuotaId = str(timestamp)

        def make_row(shard, timestamp, chunk, part, spec):
            return {
                'Stream': shard,
                'Snapshot': str(timestamp),
                'Table': '',
                'ChunkId': YsonUint64(chunk),
                'PartId': YsonUint64(part),
                'Resource': spec.SerializeToString()
            }

        if 'remote_storage' in oth:  # lrc
            chunk, part = map(int, oth.split('/')[1:])
            spec.Labels.Label["ChunkId"] = str(chunk)
            spec.Labels.Label["PartId"] = str(part)
            spec.Replication = 1
            spec.MinAliveReplica = 1
            if part < 12:
                spec.BalancingWeight = 1
            elif part < 14:
                spec.BalancingWeight = 0.5
            else:
                spec.BalancingWeight = 0.1
            yield make_row(shard, timestamp, chunk, part, spec)
        elif 'multi_level_cache_replicas' in oth:  # replicas
            chunk, part = map(int, oth.split('/')[1:])
            spec.Labels.Label["ChunkId"] = str(chunk)
            spec.Replication = generator.replicas_parts_replication
            spec.MinAliveReplica = generator.replicas_parts_replication - 1
            spec.BalancingWeight = 0.001
            assert part == 0
            yield make_row(shard, timestamp, chunk, part, spec)

    merged_inputs = list(torrent_tables_generator(mr_prefix, enable_merged=False))
    yt_client.run_reduce(reducer, merged_inputs, spec_tmp_part, reduce_by='Name')
    yt_client.run_sort(spec_tmp_part, "<append=%true>" + SPECS_TABLE, sort_by=['Stream', 'Snapshot', 'Table', 'ChunkId', 'PartId'])
    yt_client.remove(spec_tmp_part)
    yt_client.create('map_node', spec_merged_marker)


class ChunkCtlSpecsController(sdk.Controller):
    def __init__(self, observer, yt_client, readonly):
        self._observer = observer
        self._yt_client = yt_client
        super(ChunkCtlSpecsController, self).__init__()
        self._readonly = readonly

    def execute(self):
        if not self._readonly:
            for generation in self._observer.get_last_generations(2):
                prepare_chunk_ctl_specs(generation)

    def __str__(self):
        return '{}({})'.format(self.__class__.__name__, self._observer.tier.name)


def prepare_jupiter_tier1_torrents(timestamp, target, tier_name, yt_client=yt):
    mr_prefix = '{}/{}'.format(JUPITER_MRPREFIX, timestamp_to_yt_state(timestamp))
    merged_inputs = list(torrent_tables_generator(mr_prefix))
    now = datetime.datetime.now().isoformat()

    def reducer(key, rows):
        for row in rows:
            shard, oth = key['Name'].split('/', 1)

            if tier_name not in shard:
                continue

            suffix_map = {
                0: '/dynamic',
                1: '/pudge',
                2: '',
            }
            name = '{}-{}/{}{}'.format(shard, timestamp, oth, suffix_map[row['@table_index'] % 3])
            yield {
                'name': name,
                'rbtorrent': row['Rbtorrent'],
                'time': now,
                'size': 0
            }

    tmp_table = yt.TablePath(target) + '.tmp'
    yt_client.run_reduce(reducer, merged_inputs, tmp_table, reduce_by='Name', format=yt.YsonFormat(control_attributes_mode="row_fields"))
    yt_client.run_sort(tmp_table, yt.TablePath(target, schema=NamespaceTable.schema), sort_by=['name'])
    yt_client.remove(tmp_table)


def prepare_jupiter_plain_torrents(timestamp, target, torrents_table_name, tier_name, yt_client=yt):
    input_table = '{}/{}/{}'.format(JUPITER_MRPREFIX, timestamp_to_yt_state(timestamp), torrents_table_name)
    now = datetime.datetime.now().isoformat()

    def reducer(key, rows):
        for row in rows:
            if tier_name not in key['Name']:
                continue

            # plain torrents have names in Jupiter shard format, i.e. '{tier}/{group}-{index}'
            name = '{}-{}'.format(key['Name'].replace('/', '-'), timestamp)
            yield {
                'name': name,
                'rbtorrent': row['Rbtorrent'],
                'time': now,
                'size': 0
            }
    yt_client.run_reduce(reducer, input_table, yt.TablePath(target, schema=NamespaceTable.schema), reduce_by='Name')


def ensure_path(yt_client, path):
    if not yt_client.exists(path):
        ensure_path(yt_client, path.rsplit('/', 1)[0])
        yt_client.create('map_node', path)


def make_dynamic(yt_client, path):
    if yt_client.exists(path):
        if not yt_client.get(path + '/@dynamic'):
            yt_client.alter_table(path, dynamic=True)
        if yt_client.get(path + '/@tablet_state') == 'unmounted':
            yt_client.set(path + '/@tablet_cell_bundle', 'cajuper')
            yt_client.set(path + '/@primary_medium', 'ssd_blobs')
            yt_client.mount_table(path)


class JupiterCopierController(sdk.Controller):
    def __init__(self, observer, namespace_prefix, yt_client, readonly):
        self._observer = observer
        self._yt_client = yt_client
        self._namespace_prefix = namespace_prefix
        super(JupiterCopierController, self).__init__()
        self._readonly = readonly

    def __str__(self):
        return '{}({})'.format(self.__class__.__name__, self._observer.tier.name)

    def _prepare_torrents(self, timestamp, output_path, yt_client):
        raise NotImplementedError()

    def execute(self):
        for generation in self._observer.get_last_generations(2):
            namespace = '{}/{}/{}'.format(self._namespace_prefix, generation, self._observer.tier.name)
            path = namespace_to_path(namespace)
            if not self._readonly:
                ensure_path(self._yt_client, path.rsplit('/', 1)[0])
            if not self._yt_client.exists(path):
                try:
                    _log.info('create namespace for yt copier %s', namespace)
                    if self._readonly:
                        continue
                    self._prepare_torrents(generation, path, self._yt_client)
                except Exception as e:
                    _log.exception(e)
            if not self._readonly:
                make_dynamic(self._yt_client, path)


class JupiterChunksCopierController(JupiterCopierController):
    def __init__(self, observer, namespace_prefix, yt_client, readonly):
        super(JupiterChunksCopierController, self).__init__(observer, namespace_prefix, yt_client, readonly)

    def _prepare_torrents(self, timestamp, output_path, yt_client):
        prepare_jupiter_tier1_torrents(timestamp, output_path, self._observer.tier.name, yt_client)


class JupiterPlainCopierController(JupiterCopierController):
    def __init__(self, observer, namespace_prefix, yt_client, readonly, torrents_table_name):
        self._torrents_table_name = torrents_table_name
        super(JupiterPlainCopierController, self).__init__(observer, namespace_prefix, yt_client, readonly)

    def _prepare_torrents(self, timestamp, output_path, yt_client):
        prepare_jupiter_plain_torrents(timestamp, output_path, self._torrents_table_name, self._observer.tier.name, yt_client)
