import logging

import infra.callisto.controllers.multibeta.slots as multibeta_slots
import infra.callisto.controllers.multibeta.tables as multibeta_tables
import infra.callisto.controllers.deployer.controller as deploy_controller
import infra.callisto.controllers.slots.controllers as slots_controllers
import infra.callisto.controllers.slots.slot as slots_slot
import infra.callisto.controllers.slots.state as slots_state
import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.sdk.registry as registry
import infra.callisto.controllers.sdk.tier as tiers
import infra.callisto.controllers.utils.gencfg_api as gencfg_api
import infra.callisto.libraries.yt as yt_utils
import infra.callisto.controllers.utils.funcs as funcs_utils

import configuration
import conf_generator
import oldprod2
import slots
import tables

Tag = 'stable-154-r253'

DEPLOYER_GROUP = gencfg_api.GencfgGroup('SAS_WEB_REFRESH_UNITED_3DAY_FRA1_BASE_DEPLOY', Tag, mtn=False)

RTYSERVER_SLOTS = {
    ('saas_frozen_beta1',): {
        tiers.SaasFrozen3dayTier: gencfg_api.GencfgGroup('SAS_WEB_REFRESH_UNITED_3DAY_FRA1_BASE', Tag, mtn=False),
    },
    ('saas_frozen_beta2',): {
        tiers.SaasFrozen3dayTier: gencfg_api.GencfgGroup('SAS_WEB_REFRESH_UNITED_3DAY_FRA2_BASE', Tag, mtn=False),
    },
}

MMETA_AGENTS = {
    'saas_frozen_beta1': gencfg_api.GencfgGroup('SAS_WEB_REFRESH_UNITED_3DAY_FRA1_MMETA', Tag, mtn=False),
    'saas_frozen_beta2': gencfg_api.GencfgGroup('SAS_WEB_REFRESH_UNITED_3DAY_FRA2_MMETA', Tag, mtn=False),
}

SLOTS = MMETA_AGENTS.keys()


def get_target_table(readonly=True, yt_client=None):
    yt_client = yt_client or yt_utils.create_yt_client('arnold')
    return multibeta_tables.TargetTable(
        yt_client, '//home/cajuper/user/saas/multibeta/base/target', configuration.BaseBetaConfiguration, readonly,
    )


def get_configs_table(readonly=True, yt_client=None):
    yt_client = yt_client or yt_utils.create_yt_client('arnold')
    return tables.ConfigsTable(yt_client, '//home/cajuper/user/saas/multibeta/configs', readonly)


class Ctrl(sdk.Controller):
    @property
    def path(self):
        return self._name

    def __init__(self, name, slots_table, slots_ctrl, deployers_proxy, conf_gen, state_table):
        super(Ctrl, self).__init__()
        self._name = name
        self._slots_table = slots_table
        self._slots_ctrl = slots_ctrl
        self._deployers = deployers_proxy
        self._conf_gen = conf_gen
        self._state_table = state_table

        self.register(slots_ctrl, deployers_proxy._deploy_ctrl, conf_gen)
        self.add_handler('/ready_commits', self._ready_commits)

    def execute(self):

        target_states = set()

        last_state = self._state_table.last_state()
        target_states.add(slots_state.SlotState(funcs_utils.yt_state_to_timestamp(last_state['State'])))
        logging.debug('Last deploy state %s', target_states)

        for slot_id in self._slots_ctrl.slots_ids:
            configurations = self._slots_table.top_n(slot_id, 1)
            target_states.update(set(slots_state.SlotState(config.index_timestamp) for config in configurations))
            logging.debug('%s slot states %s', slot_id, target_states)

        self._deployers.target_states = target_states

        logging.debug('Observed states %s', self._deployers.observed_states)

        if self._deployers.observed_states:
            new_deployed_state = sorted(self._deployers.observed_states).pop()
            logging.debug('Set state for searchers %s', new_deployed_state)
            self._conf_gen.set_target_index_state(new_deployed_state)

            for slot_id in self._slots_ctrl.slots_ids:
                configurations = self._slots_table.top_n(slot_id, 1)
                self._slots_ctrl.set_slot_target(slot_id, configurations)

    def _ready_commits(self):
        return {
            'slots': [
                {
                    'slot_id': slot_id,
                    'ready_commits': sorted(self._slots_ctrl.ready_revisions(slot_id)),
                }
                for slot_id in sorted(self._slots_ctrl.slots_ids)
            ]
        }


def _make_agent_shard_mapping():
    return gencfg_api.get_agent_shard_number_mapping([
        group
        for descr in RTYSERVER_SLOTS.itervalues()
        for group in descr.itervalues()
    ])


def make_ctrl(readonly):
    yt_client = yt_utils.create_yt_client('arnold')
    config_table = get_configs_table(readonly=True, yt_client=yt_client)
    target_table = get_target_table(readonly, yt_client)

    return Ctrl(
        'SaasFrozenBeta',
        target_table,
        multibeta_slots.SlotsCtrl(
            slots.make_rtyserver_ctrls(RTYSERVER_SLOTS) +
            slots.make_mmeta_ctrls(MMETA_AGENTS)
        ),

        slots_controllers.DeployerProxy(
            slots_slot.Slot('vla-slot', tiers.SaasFrozen3dayTier, None, None),  # BASE_GROUP, Tag),
            deploy_controller.make_deployer_controller2(
                [DEPLOYER_GROUP],
                download_args_callback=lambda resource, host: {'max_dl_speed': '25M'}
            ),
            _make_agent_shard_mapping(),
        ),

        conf_generator.BaseBetaGenerator(
            oldprod2.get_status_table(),
            tables.YappyTable(yt_client, '//home/yappy/saas_base_controllers'),
            config_table,
            target_table,
            SLOTS,
        ),
        state_table=tables.IndexStateTable(yt_client, '//home/saas/frozen_shards/SaasFrozen3dayTier0'),
    )


registry.register('saas/betas/frozen', make_ctrl, [registry.ReportsBackends.V2.sas])
