import collections
import datetime
import logging

import oldprod2
import yt_observers
import yt_copier
import pip
# import man
import sas
import vla
import tables

import infra.callisto.controllers.build.innerbuilder as innerbuilder
import infra.callisto.controllers.slots.extctrl as extctrl
import infra.callisto.controllers.sdk as sdk
import infra.callisto.controllers.sdk.tier as tier
import infra.callisto.controllers.sdk.registry as registry
import infra.callisto.controllers.utils.gencfg_api as gencfg_api
import infra.callisto.libraries.yt as yt_utils


class Ctrl(sdk.Controller):
    """
    Main video conveyor: build, deploy to pip and prod; and multibetas.
    """
    path = 'video'

    def __init__(self, readonly):
        super(Ctrl, self).__init__()

        self.readonly = readonly

        builders_default_topology = 'stable-157-r974'
        self.builders = {
            tier.VideoPlatinum: innerbuilder.make_controller(
                yt_observers.Platinum,
                instance_provider=gencfg_api.InstanceProvider(
                    [gencfg_api.GencfgGroup('VLA_VIDEO_PLATINUM_BUILD', builders_default_topology, mtn=True)],
                    report_tags={'VLA_VIDEO_PLATINUM_BUILD'}
                ),
                space_needed_full_build_coeff=1.0,
                space_needed_inc_build_coeff=1.0,
                assist_last_n_on_finish=20,
                remove_failed_tasks=False,
            ),
            tier.VideoTier0: innerbuilder.make_controller(
                yt_observers.Tier0,
                instance_provider=gencfg_api.InstanceProvider(
                    [gencfg_api.GencfgGroup('VLA_VIDEO_TIER0_BUILD', builders_default_topology, mtn=True)],
                    report_tags={'VLA_VIDEO_TIER0_BUILD'}
                ),
                space_needed_full_build_coeff=1.0,
                space_needed_inc_build_coeff=1.0,
                assist_last_n_on_finish=20,
                remove_failed_tasks=False,
            ),
            tier.VideoEmbeddingPlatinum: innerbuilder.make_controller(
                yt_observers.EmbeddingPlatinum,
                instance_provider=gencfg_api.InstanceProvider(
                    [gencfg_api.GencfgGroup('VLA_VIDEO_PLATINUM_EMBEDDING_INDEX_BUILD', builders_default_topology, mtn=True)],
                    report_tags={'VLA_VIDEO_PLATINUM_EMBEDDING_INDEX_BUILD'}
                ),
                space_needed_full_build_coeff=1.0,
                space_needed_inc_build_coeff=1.0,
                assist_last_n_on_finish=2,
                remove_failed_tasks=False,
            ),
            tier.VideoEmbeddingTier0: innerbuilder.make_controller(
                yt_observers.EmbeddingTier0,
                instance_provider=gencfg_api.InstanceProvider(
                    [gencfg_api.GencfgGroup('VLA_VIDEO_TIER0_EMBEDDING_INDEX_BUILD', builders_default_topology, mtn=True)],
                    report_tags={'VLA_VIDEO_TIER0_EMBEDDING_INDEX_BUILD'}
                ),
                space_needed_full_build_coeff=1.0,
                space_needed_inc_build_coeff=1.0,
                assist_last_n_on_finish=2,
                remove_failed_tasks=False,
            ),
            tier.VideoInvertedPlatinum: innerbuilder.make_controller(
                yt_observers.InvertedPlatinum,
                instance_provider=gencfg_api.InstanceProvider(
                    [gencfg_api.GencfgGroup('VLA_VIDEO_PLATINUM_INVERTED_INDEX_BUILD', builders_default_topology, mtn=True)],
                    report_tags={'VLA_VIDEO_PLATINUM_INVERTED_INDEX_BUILD'}
                ),
                space_needed_full_build_coeff=1.0,
                space_needed_inc_build_coeff=1.0,
                assist_last_n_on_finish=2,
                remove_failed_tasks=False,
            ),
            tier.VideoInvertedTier0: innerbuilder.make_controller(
                yt_observers.InvertedTier0,
                instance_provider=gencfg_api.InstanceProvider(
                    [gencfg_api.GencfgGroup('VLA_VIDEO_TIER0_INVERTED_INDEX_BUILD', builders_default_topology, mtn=True)],
                    report_tags={'VLA_VIDEO_TIER0_INVERTED_INDEX_BUILD'}
                ),
                space_needed_full_build_coeff=1.0,
                space_needed_inc_build_coeff=1.0,
                assist_last_n_on_finish=2,
                remove_failed_tasks=False,
            ),
        }

        self._state_table = tables.get_custom_prod_state_table(readonly)
        self._deploy_ruler = make_deploy_ruler()
        self._switcher = make_switcher(self.readonly)
        self._notifications = []

        self._prod_state_table = oldprod2.get_status_table(readonly=True)
        self.pip = extctrl.ExternalSlotsController(_ctrls + '/video/pip', readonly, pip.get_yt_target_table(readonly), pip.get_yt_status_table(readonly=True))
        # self.man = extctrl.ExternalSlotsController(_ctrls + '/video/prod-man', readonly, man.get_yt_target_table(readonly), man.get_yt_status_table(readonly=True))
        self.sas = extctrl.ExternalSlotsController(_ctrls + '/video/prod-sas', readonly, sas.get_yt_target_table(readonly), sas.get_yt_status_table(readonly=True))
        self.vla = extctrl.ExternalSlotsController(_ctrls + '/video/prod-vla', readonly, vla.get_yt_target_table(readonly), vla.get_yt_status_table(readonly=True))

        yt_client = yt_utils.create_yt_client('arnold')
        self.yt_copiers = {
            tier.VideoPlatinum: yt_copier.VideoChunksCopierController(
                yt_observers.Platinum,
                'video/prod/yt',
                yt_client,
                readonly
            ),
            "chunk-ctl-supplier": yt_copier.ChunkCtlSpecsController(
                yt_observers.Platinum,
                yt_client,
                readonly
            )
        }

        self.register(
            self.pip,
            # self.man,
            self.sas,
            self.vla,
            *(self.yt_copiers.values() + self.builders.values())
        )

    def execute(self):
        if self._prod_state_table.common_state:
            prod_timestamp = int(self._prod_state_table.common_state)

            if 20 <= datetime.datetime.now().hour < 21:
                self._set_prod(prod_timestamp)

            self._set_pip(prod_timestamp)
            self._notifications = self._switcher.check_and_switch(prod_timestamp)

            self._write_state_to_yt(prod_timestamp)

        else:
            logging.debug('No common state observed, leaving pip & prod intact')

    def _common_builders_timestamps(self):
        builders = self.builders.values()
        timestamps = set(builders[0].timestamps())
        for builder in builders:
            timestamps &= set(builder.timestamps())
        return timestamps

    def _set_pip(self, prod_timestamp):
        deploy_target = self._deploy_ruler.to_deploy_on_acceptance(prod_timestamp)
        basesearch_target = self._deploy_ruler.search_target_on_acceptance(deploy_target, prod_timestamp)
        self.pip.set_target_state(deploy_target, basesearch_target, null_if_not_deployed=True)

    def _write_state_to_yt(self, prod_timestamp):
        ts, yt_timestamp = self._state_table.head()
        if yt_timestamp != prod_timestamp:
            self._state_table.write(prod_timestamp)
            logging.warning('Write to state table %s -> %s', yt_timestamp, prod_timestamp)

    def _set_prod(self, prod_timestamp):
        deploy_target = self._deploy_ruler.to_deploy_on_prod(prod_timestamp)
        assert prod_timestamp in deploy_target
        # for index, location in enumerate((self.sas, self.vla, self.man)):
        for index, location in enumerate((self.sas, self.vla)):
            if index * 15 < datetime.datetime.now().minute < (index + 1) * 15:
                location.set_target_state(deploy_target, prod_timestamp)

    def json_view(self):
        result = collections.OrderedDict()
        result['__readonly__'] = self.readonly
        # result['man'] = self.man.json_view()
        result['sas'] = self.sas.json_view()
        result['vla'] = self.vla.json_view()
        result['pip'] = self.pip.json_view()
        result['prod'] = {}
        result['builders'] = {
            tier_.name: builder.json_view() for tier_, builder in self.builders.items()
        }
        return result

    def build_progress(self, tier_name):
        for _tier in self.builders:
            if _tier.name == tier_name:
                return self.builders[_tier].build_progress(tier_name)

    def notifications(self):
        return self._notifications


_ctrls = 'http://ctrl.clusterstate.yandex-team.ru'


def get_acceptance_table():
    return sdk.prod_ruler.AcceptanceTableSource(
        path='//home/videoindex/full/index/shard_deploy/acceptance',
        yt_client=yt_utils.create_yt_client('arnold')
    )


def make_deploy_ruler():
    return sdk.prod_ruler.DeployRuler(
        build_source=yt_observers.Sources.VideoProd,
        acceptance_source=get_acceptance_table(),
        max_states_to_keep=2,
    )


def make_search_ruler():
    return sdk.prod_ruler.SearchRuler(
        acceptance_source=get_acceptance_table(),
        target_tables=[
            location.get_yt_target_table(readonly=True)
            # for location in (man, sas, vla)
            for location in (sas, vla)
        ],
        status_tables=[
            location.get_yt_status_table(readonly=True)
            # for location in (man, sas, vla)
            for location in (sas, vla)
        ],
    )


def make_switcher(readonly):
    return sdk.prod_ruler.Switcher(
        search_ruler=make_search_ruler(),
        task_type='SWITCH_VIDEO_DB_INDEX',
        vertical_name='video',
        should_be_deployed_time=datetime.time(hour=22),
        enable_autorelease=True,
        readonly=readonly,
    )


def make_ctrl(readonly):
    return Ctrl(readonly)


registry.register('video/prod/main', make_ctrl, registry.ReportsBackends.V2.all)
