import logging

import infra.callisto.controllers.multibeta.generator as generator

import configuration


class BaseBetaGenerator(generator.GeneratorProdTable):
    def _eval_target(self, yappy_conf, prod_conf, configs):
        return configuration.BaseBetaConfiguration(
            yappy_conf['revision'],
            _eval_mmeta(yappy_conf, prod_conf, configs),
            _eval_base(yappy_conf, prod_conf, configs),
        )

    def _prod_common_config(self):
        cfg = self._prod_table.last_status()
        return _convert_prod_common_config_to_yappy(cfg.config) if cfg.config else {}


class MmetaBetaGenerator(generator.GeneratorProdTable):
    def _eval_target(self, yappy_conf, prod_conf, configs):
        return configuration.MmetaBetaConfiguration(
            yappy_conf['revision'],
            _eval_mmeta(yappy_conf, prod_conf, configs),
        )

    def _prod_common_config(self):
        cfg = self._prod_table.last_status()
        return _convert_prod_common_config_to_yappy(cfg.config) if cfg.config else {}


class QuickBetaGenerator(generator.GeneratorProdTable):
    def __init__(self, *args, **kwargs):
        super(QuickBetaGenerator, self).__init__(*args, **kwargs)
        self._target_index_state = None

    def _eval_target(self, yappy_conf, prod_conf, configs):
        return configuration.QuickBetaConfiguration(
            yappy_conf['revision'],
            str(self._target_index_state.timestamp),
            _eval_quick_base(yappy_conf, prod_conf, configs),
        )

    def _prod_common_config(self):
        cfg = self._prod_table.last_status()
        return _convert_prod_common_config_to_yappy(cfg.config) if cfg.config else {}

    def set_target_index_state(self, target_state):
        self._target_index_state = target_state

    def execute(self):
        if self._target_index_state:
            super(QuickBetaGenerator, self).execute()
        else:
            logging.warning('Skip execute due to target_index_state is %s', self._target_index_state)


def _eval_mmeta(yappy_conf, prod_conf, configs):
    return configuration.MmetaConfiguration(
        generator.get_resource('mmeta.executable', yappy_conf, prod_conf),
        generator.get_resource('mmeta.models', yappy_conf, prod_conf),
        configs['mmeta.config'],
        configs['mmeta.config_path'],
    )


def _eval_base(yappy_conf, prod_conf, configs):
    return configuration.BasesearchConfiguration(
        generator.get_resource('basesearch.executable', yappy_conf, prod_conf),
        generator.get_resource('basesearch.models', yappy_conf, prod_conf),
        configs['basesearch.config'],
        configs['basesearch.config_path'],
    )


def _eval_quick_base(yappy_conf, prod_conf, configs):
    return configuration.RtyserverConfiguration(
        generator.get_resource('quick_base.executable', yappy_conf, prod_conf),
        generator.get_resource('quick_base.models', yappy_conf, prod_conf),
        generator.get_resource('quick_base.static_models', yappy_conf, prod_conf),
        generator.get_resource('quick_base.config', yappy_conf, prod_conf),
    )


def _convert_prod_common_config_to_yappy(cfg):
    base, mmeta, quick_base = cfg['base'], cfg['mmeta'], cfg['quick_base']

    return {
        'basesearch.executable': base['httpsearch'],
        'basesearch.models': base['models.archive'],

        'mmeta.executable': mmeta['httpsearch'],
        'mmeta.models': mmeta['models.archive'],

        'quick_base.executable': quick_base['httpsearch'],
        'quick_base.models': quick_base['models.archive'],
        'quick_base.static_models': quick_base['static_models'],
        'quick_base.config': quick_base['config'],
    }
