import datetime
import functools
import logging

import infra.callisto.controllers.search_source.controller as search_source
import infra.callisto.controllers.deployer2.controller as deploy_controller
import infra.callisto.controllers.slots as slots
import infra.callisto.controllers.sdk.tier as tier
import infra.callisto.controllers.sdk.notify as notify
import infra.callisto.controllers.sdk.registry as registry
import infra.callisto.controllers.utils.sandbox_utils as sandbox_utils
import infra.callisto.libraries.yt as yt_utils

from infra.callisto.controllers.user.jupiter.chunks.external import Controller as ext_chunks_controller

import tables
import sandbox_tasks
import tier_chunks


default_topology = 'stable-158-r2285'
deployer_groups = (
    ('VLA_VIDEO_DEPLOY_PIP', default_topology),
)

LOCATION = 'pip'
NAMESPACE_PREFIX = '/video/prod/'
NAMESPACE_PREFIX_YT = '/video/prod/yt'
SUBRESOURCES = ('video',)

PLATINUM_BASESEARCH_GROUP = 'VLA_VIDEO_PLATINUM_BASE_PIP'
PLATINUM_REMOTE_STORAGE_GROUP = 'VLA_VIDEO_REMOTE_STORAGE_BASE_PIP'


def slots_(topology=default_topology):
    return (
        slots.Slot('Platinum', tier.VideoPlatinum, PLATINUM_BASESEARCH_GROUP, topology, use_mtn=True),
        slots.Slot('PlatinumEmbedding', tier.VideoEmbeddingPlatinum, 'VLA_VIDEO_PLATINUM_EMBEDDING_BETA', topology, use_mtn=True),
        slots.Slot('PlatinumInverted', tier.VideoInvertedPlatinum, 'VLA_VIDEO_PLATINUM_INVERTED_INDEX_BETA', topology, use_mtn=True),
        slots.Slot('Tier0', tier.VideoTier0, 'VLA_VIDEO_TIER0_BASE_PIP', topology, use_mtn=True),
        slots.Slot('Tier0Embedding', tier.VideoEmbeddingTier0, 'VLA_VIDEO_TIER0_EMBEDDING_BETA', topology, use_mtn=True),
        slots.Slot('Tier0Inverted', tier.VideoInvertedTier0, 'VLA_VIDEO_TIER0_INVERTED_INDEX_BETA', topology, use_mtn=True),
    )


def get_yt_status_table(readonly):
    yt_client = yt_utils.create_yt_client('arnold', use_rpc=True)
    return search_source.get_yt_status_table(yt_client, '/video/pip/status', readonly)


def get_yt_target_table(readonly):
    yt_client = yt_utils.create_yt_client('arnold', use_rpc=True)
    return search_source.get_yt_target_table(yt_client, 'video/pip/target_v2', readonly)


class PipNotReadyNotification(notify.ValueNotification):
    name = 'pip-not-ready'
    message_template = 'pip search_target is not observed (elapsed {value} seconds)'
    ranges = (
        notify.Range(notify.NotifyLevels.IDLE, None, 3600),
        notify.Range(notify.NotifyLevels.WARNING, 3600, None),
    )


class PipController(search_source.YtDrivenSourceController):
    def __init__(self, name, slots_, deployer, target_table, status_table, custom_yt_state_table, readonly=True):
        super(PipController, self).__init__(name, slots_, deployer, target_table, status_table)
        self.readonly = readonly
        self._custom_yt_state_table = custom_yt_state_table
        self._notifications = notify.NotificationsAggregator()

    def notifications(self):
        super_notifications = super(PipController, self).notifications()
        since_target_modified = int((datetime.datetime.now() - self._target_modification_time).total_seconds())
        if (
            self._target_modification_time != datetime.datetime.min
            and self._searcher_target_state
            and self._searcher_target_state != self.get_observed_state()[1]
        ):
            return super_notifications + [PipNotReadyNotification(since_target_modified)] + self._notifications.get_notifications()
        return super_notifications + [PipNotReadyNotification(0)] + self._notifications.get_notifications()

    def execute(self):
        super(PipController, self).execute()
        self._write_custom_state_to_yt()
        self._run_switch_pip_configs_task(self._searcher_target_state.timestamp)
        self._notifications.push_buffer()

    def _write_custom_state_to_yt(self):
        deployers_states, searchers_state = self.get_observed_state()
        ts, yt_state = self._custom_yt_state_table.head()
        if searchers_state and yt_state != searchers_state.timestamp:
            self._custom_yt_state_table.write(searchers_state.timestamp)
            logging.warning('Write to state table %s -> %s', yt_state, searchers_state.timestamp)

    def _run_switch_pip_configs_task(self, db_timestamp):
        if not db_timestamp:
            return

        sb_task = sandbox_tasks.switch_pip_configs_task(db_timestamp, self.readonly)

        if not sb_task:
            logging.debug('Switch PIP Task is not created (MODE: %s)', self.readonly)
        elif sandbox_utils.is_task_failed(sb_task):
            logging.debug('There is failed switch PIP task id %s', sb_task['id'])
            self._notifications.add_notification(notify.TextNotification(
                'Check switch PIP task {} and run it manually'.format(sb_task['url']),
                notify.NotifyLevels.WARNING,
            ))
        else:
            logging.debug('Switch PIP task found %s', sb_task['url'])


def make_controller(readonly):
    deploy_ctrl = deploy_controller.make_controller(
        deployer_groups,
        download_args_callback=lambda resource, host: {'max_dl_speed': '100M', 'hardlink': True},
        mtn=True
    )

    slot_ctrls = search_source.make_slot_controllers(
        slots_(),
        deploy_ctrl,
        namespace_prefix=NAMESPACE_PREFIX
    )

    slot_ctrls['platinum_chunks'] = tier_chunks.make_chunks_ctrl(
        location=LOCATION,
        deploy_ctrl=deploy_ctrl,
        namespace_prefix=NAMESPACE_PREFIX_YT,
        remote_storage_group=(PLATINUM_REMOTE_STORAGE_GROUP, default_topology),
        readonly=readonly,
        subresources=SUBRESOURCES,
        configs_generator=functools.partial(
            sandbox_tasks.generate_rs_configs_task,
            location=LOCATION,
            topology=default_topology,
            tier=tier.VideoPlatinum,
            namespace_prefix=NAMESPACE_PREFIX_YT,
            subresources=SUBRESOURCES,
            rs_group=PLATINUM_REMOTE_STORAGE_GROUP,
            basesearch_groups=[PLATINUM_BASESEARCH_GROUP],
            readonly=readonly
        ),
        tier=tier.VideoPlatinum,
    )
    slot_ctrls['platinum_chunks_ext'] = ext_chunks_controller(
        '//home/cajuper/user/video/prod/chunks/ctl/pip/chunks_status',
        '//home/cajuper/user/video/prod/chunks/ctl/pip/chunks_target',
        tier=tier.VideoPlatinum,
        optional=False,
        readonly=readonly,
    )

    return PipController(
        name='pip_video',
        slots_=slot_ctrls,
        deployer=deploy_ctrl,
        target_table=get_yt_target_table(readonly),
        status_table=get_yt_status_table(readonly),
        custom_yt_state_table=tables.get_custom_pip_state_table(readonly),
        readonly=readonly,
    )


registry.register('video/prod/acceptance', make_controller, [registry.ReportsBackends.V2.vla])
