import infra.callisto.controllers.sdk.tier as tiers
import infra.callisto.controllers.utils.entities as entities
import infra.callisto.controllers.utils.gencfg_api as gencfg_api

from infra.callisto.controllers.user.jupiter.chunks.controller import SlotCtrlAdapter
from infra.callisto.controllers.user.jupiter.chunks.allocator import Allocator

CHUNK_NAMESPACE_PREFIX = '/video/prod/chunks'


class RemoteStorageChunksGenerator(object):
    def __init__(self, tier=tiers.VideoPlatinum, chunk_size_override=None):
        self.tier = tier
        self.size_override = chunk_size_override

        self.erasure_chunks_cnt = 13
        self.erasure_parts_cnt = 16

        self.cache_chunks_cnt = 13
        self.cache_parts_cnt = 1

        self.replicas_chunks_cnt = 13
        self.replicas_parts_cnt = 1
        self.replicas_parts_replication = 3

        self.chunks_size = 0.01 * (1024 ** 3)
        self.replicas_chunks_size = 0.01 * (1024 ** 3)

    def shards_count(self):
        return self.tier.shards_count

    def list_erasure_chunks(self, generation):
        lst = []
        for shard in self.tier.list_shards(generation):
            for i in self.erasure_chunk_ids():
                for j in xrange(self.erasure_parts_cnt):
                    lst.append(entities.Chunk(
                        shard,
                        'remote_storage/{}/{}'.format(i, j),
                    ))
        return lst

    def list_replicas_chunks(self, generation):
        lst = []

        for shard in self.tier.list_shards(generation):
            for chunk_num in self.replicas_chunk_ids():
                for part_num in xrange(self.replicas_parts_cnt):
                    lst.append(entities.Chunk(
                        shard,
                        'multi_level_cache_replicas/{}/{}'.format(chunk_num, part_num),
                    ))

        return lst

    def chunk_size(self, chunk):
        if self.size_override is None:
            if chunk.number in self.replicas_chunk_ids():
                return self.replicas_chunks_size
            return self.chunks_size
        else:
            return self.size_override

    def erasure_chunk_ids(self):
        return xrange(0, self.erasure_chunks_cnt)

    def cache_chunk_ids(self):
        return xrange(self.erasure_chunks_cnt, self.erasure_chunks_cnt + self.cache_chunks_cnt)

    def replicas_chunk_ids(self):
        first_replicas_chunk_num = self.erasure_chunks_cnt + self.cache_chunks_cnt
        last_replicas_chunk_num = first_replicas_chunk_num + self.replicas_chunks_cnt
        return xrange(first_replicas_chunk_num, last_replicas_chunk_num)

    def chunk_parts_count(self, chunk):
        if chunk.number in self.erasure_chunk_ids():
            return self.erasure_parts_cnt
        elif chunk.number in self.cache_chunk_ids():
            return self.cache_parts_cnt
        elif chunk.number in self.replicas_chunk_ids():
            return self.replicas_parts_cnt
        raise ValueError('Unknown chunk with {} number'.format(chunk.number))

    def chunk_replication(self, chunk):
        if chunk.number in self.replicas_chunk_ids():
            return self.replicas_parts_replication
        return 1


def make_chunk_generator(tier, chunk_size_override=None):
    if tier == tiers.VideoPlatinum:
        return RemoteStorageChunksGenerator(chunk_size_override=chunk_size_override)

    RuntimeError('Tier {} is not supported'.format(tier))


def make_chunks_ctrl(
        location,
        deploy_ctrl,
        namespace_prefix,
        remote_storage_group,
        readonly=True,
        remote_storage_slots=None,
        generation_max_space_share=1.0,
        subresources=('',),
        chunk_size_override=None,
        configs_generator=None,
        tier=tiers.VideoPlatinum,
        **allocator_args
):
    remote_storage_host_agent_map = gencfg_api.strict_host_agent_mapping([remote_storage_group], mtn=True)
    if remote_storage_slots is None:
        remote_storage_slots = [remote_storage_group]

    remote_storage_instances = []
    for group in remote_storage_slots:
        for instance in gencfg_api.searcher_lookup_instances(*group):
            remote_storage_instances.append(instance)

    return SlotCtrlAdapter(
        name='{}Chunks'.format(tier.name),
        deploy_ctrl=deploy_ctrl,
        namespace_prefix=namespace_prefix,
        allocator=Allocator(location, tier, CHUNK_NAMESPACE_PREFIX, readonly, **allocator_args),
        chunks_generator=make_chunk_generator(tier, chunk_size_override),
        remote_storage_instances=remote_storage_instances,
        remote_storage_host_agent_map=remote_storage_host_agent_map,
        generation_max_share=generation_max_space_share,
        chunk_resources=subresources,
        configs_generator=configs_generator,
        enable_deploy_multi_level_cache_replicas=True
    )
