"""
Wrappers over remote controllers
"""

from collections import namedtuple

import requests
import gevent

import entities
import infra.callisto.controllers.sdk.tier as tier


class BuilderException(RuntimeError):
    pass


class BuilderUnavailable(BuilderException):
    pass


class SlotsControllerException(RuntimeError):
    pass


ShardState = namedtuple('ShardState', ['status'])


class ROBuilder(object):
    request_timeout = 10

    def __init__(self, url, tier_name):
        self._url = url
        self._tier_name = tier_name

    def _get(self, url, params=None):
        try:
            response = requests.get(url, timeout=self.request_timeout, params=params)
            if response.status_code == 503:
                raise BuilderUnavailable('builder restarting')
        except requests.ConnectionError:
            raise BuilderUnavailable('builder offline')
        except OSError:
            raise BuilderException('socket error')

        response.raise_for_status()

        return response.json()

    def prepared(self):
        return {
            shard for shard, state in self.state().items()
            if state.status == entities.ShardStatus.Prepared
        }

    def building(self):
        return {
            shard for shard, state in self.state().items()
            if state.status == entities.ShardStatus.Building
        }

    def state(self):
        url = '{}/view/build_progress/{}'.format(self._url, self._tier_name)
        with gevent.Timeout(seconds=self.request_timeout, exception=BuilderUnavailable):
            state = self._get(url)

        return {
            tier.parse_shard(key): ShardState(
                value['status'],
            )
            for key, value in state.items()
        }

    def timestamps(self):
        timestamps = {shard.timestamp for shard in self.state()}
        return timestamps


class SlotsController(object):
    request_timeout = 60

    def __init__(self, url, readonly=True):
        self._url = url
        self._readonly = readonly

    def _get(self, url):
        try:
            response = requests.get(url, timeout=self.request_timeout)
            if not response.ok:
                raise SlotsControllerException(response.reason)
            return response.json()
        except requests.RequestException:
            raise SlotsControllerException('request exception')
        except (ValueError, TypeError):
            raise SlotsControllerException('json decoding error')

    def _get_state(self, state_type):
        status = self._get('{}/status'.format(self._url))
        deployer_timestamps = set()
        basesearch_timestamp = None

        common_status = status['common']
        for state in filter(None, common_status['deployer'][state_type]):
            deployer_timestamps.add(state['timestamp'])
        if common_status['searcher'][state_type]:
            basesearch_timestamp = common_status['searcher'][state_type]['timestamp']

        return deployer_timestamps, basesearch_timestamp

    def observed_state(self):
        return self._get_state('observed')

    def target_state(self):
        return self._get_state('target')

    def build_progress(self, tier_name):
        url = '{}/view/build_progress/{}'.format(self._url, tier_name)
        return self._get(url)

    def deploy_progress(self, tier_name=None):
        url = '{}/view/deploy_progress'.format(self._url)
        if tier_name:
            url += '/{}'.format(tier_name)
        return self._get(url)

    def deploy_percentage(self, timestamp=None):
        url = '{}/view/deploy_percentage'.format(self._url)
        if timestamp:
            url += '/{}'.format(timestamp)
        return self._get(url)

    def searchers_state(self, slot_name=None):
        url = '{}/view/searchers_state'.format(self._url)
        if slot_name:
            url += '/{}'.format(slot_name)
        return self._get(url)[slot_name] if slot_name else self._get(url)
