from collections import namedtuple

import infra.callisto.deploy.resource as deploy_resource

from . import funcs


class HostsIntersection(RuntimeError):
    pass


HostSpec = namedtuple('HostSpec', ['host', 'freespace', 'rack'])


class ShardStatus(object):
    Building = 'BUILD'
    Prepared = 'DONE'
    NotStarted = 'IDLE'
    Failed = 'FAILURE'
    Null = 'none'


class Shard(namedtuple('Shard', ['tier', 'group_number', 'shard_number', 'timestamp'])):
    shard_template = None
    number_template = None

    def __new__(cls, tier, group_number, shard_number, timestamp):
        # noinspection PyArgumentList
        return super(Shard, Shard).__new__(cls, tier, int(group_number), int(shard_number), int(timestamp))

    @funcs.cached_property
    def number(self):
        return self.number_template.format(
            group_number=self.group_number,
            shard_number=self.shard_number,
        )

    @funcs.cached_property
    def name(self):
        return self.shard_template_without_timestamp.format(
            prefix=self.tier.prefix,
            tier_name=self.tier.name,
            group_number=self.group_number,
            shard_number=self.shard_number,
        )

    @funcs.cached_property
    def fullname(self):
        return self.shard_template.format(
            prefix=self.tier.prefix,
            tier_name=self.tier.name,
            group_number=self.group_number,
            shard_number=self.shard_number,
            timestamp=self.timestamp,
            date=self.date,
            time=self.time,
        )

    @classmethod
    def parse_number(cls, number):
        params = funcs.parse_by_template(number, cls.number_template)
        return int(params.get('group_number', 0)), int(params.get('shard_number', 0))

    @classmethod
    def parse_fullname(cls, tier, fullname):
        raise NotImplementedError()

    @funcs.cached_property
    def date(self):
        return funcs.timestamp_to_yt_state(self.timestamp).split('-')[0]

    @funcs.cached_property
    def time(self):
        return funcs.timestamp_to_yt_state(self.timestamp).split('-')[1]

    def __str__(self):
        return self.fullname

    def __repr__(self):
        return str(self)


class Chunk(namedtuple('Chunk', ['shard', 'path'])):
    """Part actually, not Chunk"""

    def __new__(cls, shard, path):
        # noinspection PyArgumentList
        return super(Chunk, cls).__new__(cls, shard,
                                         deploy_resource.normalize_resource_name(path))

    @funcs.cached_property
    def timestamp(self):
        return self.shard.timestamp

    @funcs.cached_property
    def number(self):
        return int(self.path.split('/')[1])

    @funcs.cached_property
    def part(self):
        return int(self.path.split('/')[2])


class Agent(namedtuple('Agent', ['host', 'port', 'node_name'])):
    def __new__(cls, host, port, **kwargs):
        if 'node_name' in kwargs and kwargs['node_name']:
            node_name = kwargs['node_name']
        else:
            node_name = _node_name(host)

        # noinspection PyArgumentList
        return super(Agent, cls).__new__(cls, str(host), int(port), str(node_name))

    def json(self):
        return {'host': self.host, 'port': self.port}

    @property
    def instance(self):
        return '{}:{}'.format(self.host, self.port)

    @property
    def short_host(self):
        # gencfg is not ready for mtn, basesearch config files are called mtn incompatible (man1-3456:8030.cfg)
        if 'gencfg-c' in self.host:
            return '-'.join(self.host.split('-')[:2])
        return self.host.split('.')[0]

    def __str__(self):
        return 'Agent({}, {})'.format(self.host, self.port)

    def __eq__(self, other):
        return (self.host, self.port) == (other.host, other.port)

    def __hash__(self):
        return hash((self.host, self.port))


def serialize_agents(agents):
    lst = []
    for agent in agents:
        lst.append(agent.json())
    return lst


def _node_name(fqdn):
    if 'gencfg-c' not in fqdn:
        return fqdn
    assert fqdn[:3] in ('sas', 'vla', 'man', 'yas', 'iva', 'myt'), 'Unparseable fqdn {}'.format(fqdn)
    host = fqdn.split('-')

    return '{}-{}.search.yandex.net'.format(host[0], host[1])


class AbstractInstance(namedtuple('_Instance', ['raw_data'])):
    @property
    def id(self):
        raise NotImplementedError()

    @property
    def hostname(self):
        raise NotImplementedError()

    @property
    def node_name(self):
        raise NotImplementedError()

    @property
    def port(self):
        raise NotImplementedError()

    @property
    def rack(self):
        raise NotImplementedError()

    @property
    def is_slow_network(self):
        raise NotImplementedError()

    @property
    def shard_number(self):
        raise NotImplementedError()

    @property
    def tags(self):
        raise NotImplementedError()

    @property
    def storage_size(self):
        raise NotImplementedError()

    @property
    def is_on_ssd(self):
        raise NotImplementedError()

    def get_agent(self):
        return Agent(
            host=self.hostname,
            port=self.port,
            node_name=self.node_name
        )
