import random
import logging
import datetime
import itertools
import collections

import pytz
import aniso8601


MOSCOW_TZ = pytz.timezone('Europe/Moscow')
UTC_TZ = pytz.utc


def topological_sort(root, links, reverse=False):
    visited = set()
    sorted_nodes = []

    def dfs(node):
        visited.add(node)
        for child in links[node]:
            if child not in sorted_nodes:
                assert child not in visited
                dfs(child)
        sorted_nodes.append(node)

    dfs(root)
    return sorted_nodes if not reverse else sorted_nodes[::-1]


def map_ignoring_exceptions(fn, iterable):
    return list(imap_ignoring_exceptions(fn, iterable))


def imap_ignoring_exceptions(fn, iterable):
    success_cnt, failed_cnt = 0, 0
    for item in iterable:
        # noinspection PyBroadException
        try:
            yield fn(item)
            success_cnt += 1
        except Exception:
            failed_cnt += 1
    if failed_cnt > 0:
        logging.debug('map %s failed %i of %i', fn, failed_cnt, success_cnt + failed_cnt)


def shuffled(iterable):
    lst = list(iterable)
    random.shuffle(lst)
    return lst


def iso_time_to_timestamp(iso_time):
    msk_dt = aniso8601.parse_datetime(iso_time)
    if msk_dt.tzinfo is None:
        msk_dt = MOSCOW_TZ.localize(msk_dt)
    utc_null_dt = UTC_TZ.localize(datetime.datetime(1970, 1, 1))
    return int((msk_dt - utc_null_dt).total_seconds())


def timestamp_to_yt_state(ts):
    return datetime.datetime.fromtimestamp(int(ts), tz=MOSCOW_TZ).strftime('%Y%m%d-%H%M%S')


def yt_state_to_timestamp(state):
    if state == '00000000-000000':
        return 0

    msk_dt = MOSCOW_TZ.localize(datetime.datetime.strptime(state, '%Y%m%d-%H%M%S'))
    utc_null_dt = UTC_TZ.localize(datetime.datetime(1970, 1, 1))
    return int((msk_dt - utc_null_dt).total_seconds())


def parse_by_template(string, template, splitter='-'):
    keys = template.split(splitter)
    values = string.split(splitter)
    assert len(keys) == len(values)
    result = {}
    for key, value in zip(keys, values):
        if key != value and ('{' in key or '}' in key):
            key = key.replace('{', '').replace('}', '').partition(':')[0]
            result[key] = value
    return result


def merge_dicts(*dicts):
    result = {}
    for dict_ in dicts:
        result.update(dict_)
    return result


def are_sets_intersecting(sets):
    return any(a & b for a, b in itertools.combinations(sets, 2))


def merge_dicts_no_intersection(*dicts):
    if are_sets_intersecting([set(dct) for dct in dicts]):
        raise ValueError('dicts are intersecting')
    return merge_dicts(*dicts)


class cached_property(object):
    def __init__(self, func):
        self.func = func

    def __get__(self, instance, cls=None):
        result = instance.__dict__[self.func.__name__] = self.func(instance)
        return result

    def __set__(self, instance, value):
        raise AttributeError('cannot set cached_property')


def invert_mapping(mapping):
    """ {a: {b, c}} -> {b: {a}, c: {a}} """
    result = collections.defaultdict(set)
    for key, values in mapping.iteritems():
        for value in values:
            result[value].add(key)
    return dict(result)
