import cPickle
import collections
import logging
import os
import requests

import infra.callisto.libraries.memoize as memoize
import infra.callisto.controllers.sdk.tier as tier

import entities


URL = 'http://api.gencfg.yandex-team.ru'


class GencfgAPIError(RuntimeError):
    pass


class GencfgGroupDoesNotExist(GencfgAPIError):
    def __init__(self, group, tag):
        super(GencfgGroupDoesNotExist, self).__init__('Group {}/{} does not exist'.format(group, tag))


GencfgGroup = collections.namedtuple('GencfgGroup', ['name', 'tag', 'mtn'])


class Instance(entities.AbstractInstance):
    def __new__(cls, raw_data, use_mtn=False):
        self = super(Instance, cls).__new__(cls, raw_data)
        self._use_mtn = use_mtn
        return self

    @property
    def id(self):
        return '{}:{}'.format(self.hostname, self.port)

    @property
    def hostname(self):
        if self._use_mtn:
            return _backbone_hostname(self.raw_data)
        return self.raw_data['hostname']

    @property
    def node_name(self):
        return self.raw_data['hostname']

    @property
    def port(self):
        return self.raw_data['port']

    @property
    def rack(self):
        return self.raw_data['rack']

    @property
    def is_slow_network(self):
        BANDWIDTH_1000_MBITS = 1000 + 100
        return self.raw_data['host_resources']['net'] < BANDWIDTH_1000_MBITS

    @property
    def shard_number(self):
        return _shard_name_to_number(self.raw_data['shard_name'])

    @property
    def tags(self):
        return set(self.raw_data['tags'])

    @property
    def storage_size(self):
        return storage_guarantee(self.raw_data)

    @property
    def is_on_ssd(self):
        return 'itag_copy_on_ssd' in self.tags

    @property
    def is_alive(self):
        return True  # compatibility with Yp InstanceProvider


class InstanceProvider(object):
    def __init__(self, groups, report_tags=None, tier=None, blacklist_nodes=None):
        self._groups = groups
        self._tags = report_tags
        self._blacklist_nodes = blacklist_nodes or set()
        self._tier = tier

        self._agents_instances = None

    def group_keys(self):
        return [group.name for group in self._groups]

    def _update(self):
        agents_instances = {}

        for group in self._groups:
            for instance_data in searcher_lookup_instances(group.name, group.tag):
                instance = Instance(instance_data, group.mtn)
                if instance.node_name not in self._blacklist_nodes:
                    agents_instances[instance.get_agent()] = instance

        assert len(agents_instances) > 0, 'Could not resolve groups {}'.format(self._groups)

        self._agents_instances = agents_instances

    @property
    def agents(self):
        return self.agents_instances.iterkeys()

    @property
    def agents_instances(self):
        if not self._agents_instances:
            self._update()
        return self._agents_instances

    @property
    def ids(self):
        return [
            instance.id
            for instance in self.agents_instances.values()
        ]

    @property
    def strict_host_agent_mapping(self):
        mapping = {}
        for agent in self.agents_instances:
            if agent != mapping.get(agent.node_name, agent):
                raise entities.HostsIntersection('{} conflict with {}'.format(agent, mapping[agent.node_name]))
            mapping[agent.node_name] = agent

        return mapping

    @property
    def agent_shard_number_mapping(self):
        return {
            agent: instance.shard_number
            for agent, instance in self.agents_instances.items()
        }

    @property
    def tags(self):
        if self._tags:
            return self._tags

        common_tags = set(filter(
            lambda x: not x.startswith('a_topology_'),
            self.agents_instances.values()[0].tags
        ))

        for i in self.agents_instances.values():
            common_tags.intersection_update(i.tags)

        return common_tags

    @property
    def tier(self):
        return self._tier


def searcher_lookup_instances(group, version_tag):
    if _use_local_cache:
        dirname = './.gencfg_cache'
        if not os.path.exists(dirname):
            os.makedirs(dirname)
        filename = '{}/{}-{}'.format(dirname, group, version_tag)
        if not os.path.exists(filename):
            instances = _searcher_lookup_instances(group, version_tag)
            with open(filename, 'w') as f:
                cPickle.dump(instances, f)
        with open(filename) as f:
            return cPickle.load(f)
    else:
        return _searcher_lookup_instances(group, version_tag)


@memoize.memoized
def _searcher_lookup_instances(group, version_tag):
    if version_tag == 'trunk':
        url = '{}/trunk/searcherlookup/groups/{}/instances'.format(URL, group)
        _log.warn('using trunk topology')
    else:
        url = '{}/tags/{}/searcherlookup/groups/{}/instances'.format(URL, version_tag, group)
    _log.info('get instances %s/%s', group, version_tag)
    resp = requests.get(url, timeout=60)
    if resp.status_code == 200:
        return resp.json()['instances']
    if resp.status_code == 404:
        raise GencfgGroupDoesNotExist(group, version_tag)
    else:
        raise GencfgAPIError('{}. {}'.format(resp.reason, url))


def searcher_lookup_agents(group, version_tag, mtn=False):
    agents = {}
    for instance in searcher_lookup_instances(group, version_tag):
        agent = entities.Agent(instance['hostname'] if not mtn else _backbone_hostname(instance),
                               instance['port'],
                               node_name=instance['hostname'])
        agents[agent] = instance
    return agents


def searcher_lookup_agents_instances(groups_and_tags, mtn=False):
    agents = {}
    for group, tag in groups_and_tags:
        agents.update(searcher_lookup_agents(group, tag, mtn))
    return agents


def strict_host_agent_mapping(groups_and_tags, mtn=False):
    mapping = {}
    for group, version_tag in groups_and_tags:
        for agent in searcher_lookup_agents(group, version_tag, mtn=mtn):
            if agent.node_name in mapping and mapping[agent.node_name] != agent:
                raise entities.HostsIntersection('{} conflict with {}'.format(agent, mapping[agent.node_name]))
            else:
                mapping[agent.node_name] = agent
    return mapping


def agent_shard_number_mapping(group, version_tag, mtn=False):
    return {
        agent: _shard_name_to_number(instance['shard_name'])
        for agent, instance in searcher_lookup_agents(group, version_tag, mtn=mtn).iteritems()
    }


def get_agents_instances(groups_list):
    assert all(isinstance(g, GencfgGroup) for g in groups_list)

    return {
        entities.Agent(instance['hostname'] if not group.mtn else _backbone_hostname(instance),
                       instance['port'],
                       node_name=instance['hostname']): instance
        for group in groups_list
        for instance in searcher_lookup_instances(group.name, group.tag)
    }


def get_agents(groups_list):
    return set(get_agents_instances(groups_list))


def get_strict_host_agent_mapping(groups_list):
    mapping = {}

    for agent in get_agents(groups_list):
        if agent.node_name in mapping and mapping[agent.node_name] != agent:
            raise entities.HostsIntersection('{} conflicts with {}'.format(agent, mapping[agent.node_name]))
        mapping[agent.node_name] = agent
    return mapping


def get_agent_shard_number_mapping(groups_list):
    return {
        agent: _shard_name_to_number(instance['shard_name'])
        for agent, instance in get_agents_instances(groups_list).iteritems()
    }


def storage_guarantee(instance):
    for volume in instance.get('volumes', []):
        if volume['guest_mount_point'] == '/shard_root':
            return int(volume['quota'])

    return int(instance.get('storages', {}).get('ssd', {}).get('size', 0)) * 1024 ** 3


def _shard_name_to_number(shard_name):
    shard = tier.parse_shard(shard_name)
    return shard.group_number, shard.shard_number


def _backbone_hostname(instance):
    return instance['hbf']['interfaces']['backbone']['hostname']


_use_local_cache = False


def use_local_cache():
    global _use_local_cache
    if not _use_local_cache:
        _use_local_cache = True
        _log.warning('Using gencfg cache on local filesystem')


_log = logging.getLogger(__name__)
