import json
import hashlib
import collections


def yt_schema(nested_obj):
    schema = []
    for key, type_ in nested_obj.keys_types():
        schema.append({
            'name': key,
            'type': type_.yt_type,
        })
    return schema


class TerminalObject(object):
    yt_type = None
    python_types = ()


class Integer(TerminalObject):
    yt_type = 'int64'
    python_types = (int, long)


class String(TerminalObject):
    yt_type = 'string'
    python_types = (str, bytes)


class NestedObject(object):
    """ Similar to namedtuple. Useful to dump deep wrapped objects
        Define `fields` with such format: [
            (field_name, <TerminalObject | NestedObject>),
            ...
        ]
    """
    fields = ()

    def __init__(self, *args, **kwargs):
        types = self._types()
        for (key, type_), value in zip(self.fields, args):
            if not _isinstance(value, type_):
                raise TypeError('type mismatch for key {}: expected {}, given {}'.format(key, type_, type(value)))
            self.__dict__[key] = value

        for key, value in kwargs.items():
            if not _isinstance(value, types[key]):
                raise TypeError('type mismatch for key {}: expected {}, given {}'.format(key, types[key], type(value)))
            self.__dict__[key] = value

        if not set(self.__dict__) == set(types):
            raise TypeError('not enough args')

    @classmethod
    def keys_types(cls):
        keys_types = []
        for field_name, field_type in cls.fields:
            if issubclass(field_type, NestedObject):
                for key, type_ in field_type.keys_types():
                    keys_types.append(('{}.{}'.format(field_name, key), type_))
            else:
                keys_types.append((field_name, field_type))
        return keys_types

    @classmethod
    def keys(cls):
        return [key for key, _ in cls.keys_types()]

    def dump_json_flat(self):
        return {
            key: self._get_by_key(key)
            for key in self.keys()
        }

    @classmethod
    def load_json_flat(cls, data):
        types = cls._types()
        keys = cls.keys()
        args = {}
        inners = collections.defaultdict(dict)
        for key, value in data.items():
            if key not in keys:
                continue
            if '.' not in key:
                args[key] = value
            else:
                head, _, tail = key.partition('.')
                inners[head][tail] = value
        for key, sub_args in inners.items():
            args[key] = types[key].load_json_flat(sub_args)
        return cls(**args)

    def __setattr__(self, key, value):
        raise AttributeError('Object is immutable')

    def __hash__(self):
        serialized = json.dumps(self.dump_json_flat(), sort_keys=True)
        hash_ = hashlib.md5()
        hash_.update(serialized)
        big_int = int(hash_.hexdigest(), 16)
        return big_int % (10 ** 12)

    def __eq__(self, other):
        return type(other) == type(self) and self.dump_json_flat() == other.dump_json_flat()

    def __ne__(self, other):
        return not (self == other)

    def __str__(self):
        return self.__class__.__name__

    @classmethod
    def _types(cls):
        return {key: type_ for key, type_ in cls.fields}

    def _get_by_key(self, key):
        head, _, tail = key.partition('.')
        obj = getattr(self, head)
        if isinstance(obj, NestedObject):
            return obj._get_by_key(tail)
        else:
            return obj


def _isinstance(value, type_):
    if issubclass(type_, NestedObject):
        return isinstance(value, type_)
    elif issubclass(type_, TerminalObject):
        return isinstance(value, type_.python_types + (type(None),))  # TODO: is it correct?
    raise ValueError('unknown type: {}'.format(type_))
