import logging

from sandbox.common.rest import Client as SandboxClient
from sandbox.common.proxy import OAuth
import sandbox.common.types.task as sandbox_ctt

import infra.callisto.libraries.memoize as memoize
import json

from . import secrets

sandbox_url = 'https://sandbox.yandex-team.ru/api/v1.0'


def get_sandbox_client():
    token = secrets.find_sandbox_token()
    return SandboxClient(base_url=sandbox_url, auth=OAuth(token))


@memoize.memoized
def get_task(task_id):
    sandbox = get_sandbox_client()

    tasks = sandbox.task.read(id=task_id, limit=1)
    if tasks['total'] == 1:
        return tasks['items'][0]
    return memoize.DontSave({})


def get_resource_description(resource_id):
    sandbox = get_sandbox_client()
    resource_found = sandbox.resource.read(id=resource_id, limit=1)

    if resource_found['total'] == 1:
        return resource_found['items'].pop()

    raise RuntimeError('Resource %s not found', resource_id)


@memoize.memoized
def get_task_resources(task_id, limit=50):
    sandbox = get_sandbox_client()

    resources = sandbox.resource.read(
        task_id=[task_id],
        limit=limit
    )

    task_is_done = [
        is_task_success(resource['task'])
        for resource in resources['items']
    ]

    if any(task_is_done):
        return resources['items']

    return memoize.DontSave(resources['items'])


def is_task_failed(task):
    return task['status'] in (
        sandbox_ctt.Status.Group.BREAK |
        sandbox_ctt.Status.Group.SCHEDULER_FAILURE |
        sandbox_ctt.Status.DELETED
    )


def is_task_success(task):
    return task['status'] in sandbox_ctt.Status.Group.SUCCEED


def is_task_released(task):
    if task['status'] == sandbox_ctt.Status.RELEASED:
        return True

    if task['status'] == sandbox_ctt.Status.SUCCESS and task['input_parameters'].get('auto_release', False):
        return True

    return False


def _to_unicode(d):
    return json.loads(json.dumps(d))


def get_last_task(task_type, params=None, tags=None, hints=None):
    sandbox = get_sandbox_client()
    tasks_found = sandbox.task.read(
        type=task_type,
        limit=10,
        tags=tags,
        all_tags=True,
        hints=hints,
        all_hints=True,
        order='-id'
    )

    if tasks_found['total'] > 0:
        unicoded_params = _to_unicode(params)
        for task in tasks_found['items']:
            found_task_params = {
                name: task['input_parameters'][name]
                for name in unicoded_params
                if name in task['input_parameters']
            }

            if found_task_params == unicoded_params:
                return task

    return {}


def run_task(task_type, params, description, auto_release=False, tags=None, hints=None):
    sandbox = get_sandbox_client()
    created_task = sandbox.task({'type': task_type})

    task_defs = {
        'owner': 'SEPE-PRIEMKA',
        'description': description,
        'priority': {
            'subclass': 'HIGH',
            'class': 'SERVICE'
        },
        'custom_fields': [
            {'name': name, 'value': val}
            for name, val in params.items() + [('auto_release', auto_release)]
        ],
    }

    if tags:
        task_defs['tags'] = tags
    if hints:
        task_defs['hints'] = hints

    sandbox.task[created_task['id']] = task_defs
    sandbox.batch.tasks.start.update(created_task['id'])

    return created_task


def run_only_task(task_type, params, description, auto_release=False, tags=None, hints=None, readonly=True):
    existent_task = get_last_task(task_type, params, tags, hints)
    if existent_task:
        return existent_task

    if not readonly:
        return run_task(task_type, params, description, auto_release, tags, hints)

    _log.info('Skip %s task create (params: %s, description: %s, auto_release: %s,'
              ' tags: %s, hints: %s) due to RO mode',
              task_type, params, description, auto_release, tags, hints)
    return None

_log = logging.getLogger(__name__)
