import time
import logging
import datetime

import requests


def _from_utc_to_local(utc_datetime):
    return utc_datetime.replace(tzinfo=None)


def _from_local_to_utc(date_time):
    now_timestamp = time.time()
    offset = datetime.datetime.fromtimestamp(now_timestamp) - datetime.datetime.utcfromtimestamp(now_timestamp)
    return (date_time - offset).replace(tzinfo=None)


def _utc_iso_format(date_time):
    date_time = _from_local_to_utc(date_time)
    return date_time.replace(tzinfo=None).isoformat() + 'Z'


class SolomonClient(object):
    _push_url = 'http://solomon.yandex.net/push/json'
    _get_url = 'http://api.solomon.search.yandex.net/data-api/get'
    _who = 'robot-gencfg'

    def __init__(self, project, cluster, service):
        self._project = project
        self._cluster = cluster
        self._service = service

        self._sensors = []

    def add_sensor(self, sensor_name, value, date_time):
        ts = _utc_iso_format(date_time)
        self._sensors.append({
            "labels": {
                "sensor": sensor_name,
            },
            "ts": ts,
            "value": value,
        })

    def flush(self):
        if not self._sensors:
            return
        data = {
            "commonLabels": {
                "project": self._project,
                "cluster": self._cluster,
                "service": self._service,
            },
            "sensors": self._sensors
        }
        resp = requests.post(self._push_url, json=data)
        resp.raise_for_status()
        logging.info('flushed %i sensors', len(self._sensors))
        self._sensors = []
