import logging
import mimetypes

import jinja2
import gevent
import bleach

import library.python.resource as resource

from . import reports


def _load_template_from_resources(name):
    return resource.find('/templates/{}'.format(name))


def _load_static_from_resources(name):
    return resource.find('/static/{}'.format(name))


def loop(func, interval):
    while True:
        # noinspection PyBroadException
        try:
            func()
        except Exception:
            logging.exception('Unhandled exception in loop')
        finally:
            gevent.sleep(interval)


class _AppContext(object):
    def __init__(self, states_):
        self.states = states_
        gevent.spawn(loop, states_.update, 60)


def configure_app(states_obj):
    from flask import Flask, Response

    app = Flask(__name__, static_folder='')
    app.register_blueprint(reports.reports_blueprint)
    app.jinja_env.loader = jinja2.FunctionLoader(_load_template_from_resources)

    app._context = _AppContext(states_obj)

    @app.template_filter('linkify')
    def linkify(s):
        def target_blank(attrs, new=False):
            attrs[(None, unicode('target'))] = unicode('_blank')
            return attrs

        return bleach.linkify(s, callbacks=[target_blank])

    @app.route('/static/<path:path>')
    def static_file(path):
        mime_type, _ = mimetypes.guess_type(path)
        return Response(_load_static_from_resources(path), mimetype=mime_type)

    return app
