import logging
import datetime
from collections import OrderedDict, defaultdict

import flask
import requests

import utils


reports_blueprint = flask.Blueprint('viewer', __name__)


@reports_blueprint.route('/')
@reports_blueprint.route('/generations')
def get_generations():
    if _states().callisto:
        return flask.render_template('callisto.html', data=generations_data())
    return flask.render_template('generations.html', data=generations_data())


@reports_blueprint.route('/panel')
def get_panel():
    data = generations_data()
    if _states().callisto:
        data.update(page='Panel', update_rate=15)
        return flask.render_template('callisto.html', data=data)
    data.update(page='Panel', update_rate=60)
    return flask.render_template('generations.html', data=data)


def get_builders_by_status():
    generations = {}
    builders = {}
    for tier in _states().tiers.values():
        aggr = defaultdict(lambda: defaultdict(int))
        for shard, state in tier.builder.build_state.items():
            timestamp = utils.get_generation_by_shard(shard)
            aggr[timestamp][state['status']] += 1
        for timestamp in aggr:
            generations[timestamp] = utils.timestamp_to_yt_state(timestamp)
            builders.setdefault(timestamp, {'str': utils.timestamp_to_yt_state(timestamp), 'tiers': OrderedDict()})
            builders[timestamp]['tiers'][tier.tier_name] = aggr[timestamp]
    return builders, generations


def generations_data():
    locations = OrderedDict()
    builders, generations = get_builders_by_status()

    for tier_name, tier in _states().tiers.items():
        for location_name, location in tier.locations.items():
            locations.setdefault(location_name, {'url': location.location_deploy_url, 'generations': OrderedDict()})
            deploy_replicas_state = location.deploy_replicas_state
            search_replicas_state = location.search_replicas_state
            for generation in sorted(deploy_replicas_state, reverse=True):
                generations[generation] = utils.timestamp_to_yt_state(generation)
                locations[location_name]['generations'].setdefault(generation, OrderedDict())
                locations[location_name]['generations'][generation][tier_name] = {
                    'deploy': deploy_replicas_state[generation],
                    'search': search_replicas_state[generation],
                }

    sorted_generations = OrderedDict()
    for generation in sorted(generations, reverse=True):
        sorted_generations[generation] = generations[generation]
    return {
        'page': 'Generations',
        'builders': builders,
        'locations': locations,
        'generations': sorted_generations,
        'builders_urls': {tier_name: tier.builder.viewer_url for tier_name, tier in _states().tiers.items()},
        'ctrls_state': getattr(_states().ctrls_checker, 'state', {}),
    }


@reports_blueprint.route('/build')
def get_build():
    undone_only = flask.request.args.get('undone_only') in ['yes', 'da', '1']
    builders, _ = get_builders_by_status()

    generations = {}
    tiers = {tier.tier_name: {'url': tier.builder.url} for tier in _states().tiers.values()}
    for tier in _states().tiers.values():
        for shard, state in tier.builder.build_state.items():
            generation = utils.get_generation_by_shard(shard)
            generations.setdefault(generation, {'str': utils.timestamp_to_yt_state(generation), 'tiers': {}})
            generations[generation]['tiers'].setdefault(tier.tier_name, {})
            if not undone_only or state['status'] != 'DONE':
                generations[generation]['tiers'][tier.tier_name][shard] = state

    for generation in generations.iterkeys():
        for tier in generations[generation]['tiers'].iterkeys():
            generations[generation]['tiers'][tier] = OrderedDict(sorted(
                generations[generation]['tiers'][tier].items(),
                key=lambda x: x[1]['status'] if x[1]['status'] != 'none' else 'AAA'
            ))

    data = {'page': 'Build', 'generations': generations, 'tiers': tiers, 'builders': builders}
    return flask.render_template('build.html', data=data)


@reports_blueprint.route('/logs')
def get_logs():
    builder = flask.request.args.get('builder')
    shard = flask.request.args.get('shard')

    logs_url = 'http://{0}/listdir?shard={1}'.format(builder, shard)
    try:
        logs_files = utils.retry_get_json_by_url(logs_url)
    except (RuntimeError, requests.RequestException, ValueError):
        logs_files = []
        logging.exception('Unable to get rawdata')
    files = {}
    for f in logs_files:
        files[f['name']] = {
            'size': f['size'],
            'mtime': datetime.datetime.fromtimestamp(int(f['mtime'])).strftime('%m-%d %H:%M')
        }
    logs_files = files

    return flask.render_template(
        'logs.html',
        data={
            'logs_files': logs_files,
            'shard': shard,
            'builder': builder,
        }
    )


@reports_blueprint.route('/logs_download')
def get_logs_download():
    builder = flask.request.args.get('builder')
    shard = flask.request.args.get('shard')
    file_ = flask.request.args.get('file')

    url = 'http://{0}/getcontent?shard={1}&file={2}'.format(builder, shard, file_)

    try:
        file_content = utils.retry_get_url(url)
    except (RuntimeError, requests.RequestException):
        flask.abort(500)
        logging.exception('Unable to get rawdata')
    else:
        return flask.Response(response=file_content, mimetype="text/plain")


@reports_blueprint.route('/perf')
def get_perf():
    if _states().perf:
        return flask.render_template('perf.html', data={'perf': _states().perf.tiers, 'page': 'Perf'})
    return flask.Response('<h2>no such page for this contour</h2>', status=404)


def _simplify_shard_name(shard_name):
    generation = utils.get_generation_by_shard(shard_name)
    shard_name = shard_name.replace(generation, utils.timestamp_to_yt_state(utils.get_generation_by_shard(generation)))
    shard_name = shard_name.replace('primus-', '')
    return shard_name


def _states():
    return flask.current_app._context.states
