import collections

import utils

import infra.callisto.libraries.memoize as memoize


class PartialUpdate(RuntimeError):
    message = 'some updates not succeeded'


class _Source(object):
    callisto = False

    def __init__(self, tiers, ctrls_checker=None, perf=None):
        self.tiers = tiers
        self.ctrls_checker = ctrls_checker
        self.perf = perf

    def update(self):
        partial = False
        for item in self.tiers.values() + filter(None, [self.perf, self.ctrls_checker]):
            try:
                item.update()
            except PartialUpdate:
                partial = True
        if partial:
            raise PartialUpdate()


class _CallistoSource(_Source):
    callisto = True


class _ControllersStateChecker(object):
    def __init__(self, ctrl_urls):
        self._ctrl_urls = ctrl_urls
        self.state = {key: [] for key in ctrl_urls}

    def update(self):
        partial = 0
        for ctrl_name, url in self._ctrl_urls.items():
            # noinspection PyBroadException
            try:
                self.state[ctrl_name] = utils.retry_get_json_by_url(url + '/info/notifications')
            except Exception:
                partial += 1
        if partial > 0:
            raise PartialUpdate()


class _Tier(object):
    def __init__(
            self,
            tier_name,
            builder,
            locations,
    ):
        self.tier_name = tier_name
        self.builder = builder
        self.locations = locations

    def update(self):
        partial = False
        for item in self.locations.values() + ([self.builder] if self.builder is not None else []):
            try:
                item.update()
            except PartialUpdate:
                partial = True
        if partial:
            raise PartialUpdate()


class _ReplicationState(object):
    def __init__(self):
        self.all = 0
        self.total = 0
        self._hist = collections.defaultdict(int)

    def add_shard(self, active_cnt, total_cnt):
        self.total += 1
        self.all += active_cnt / float(total_cnt)
        for cnt in range(0, active_cnt+1):
            self._hist[cnt] += 1

    @property
    def first(self):
        return self._hist[1]

    @property
    def hist(self):
        return {
            replicas: number
            for replicas, number in self._hist.items()
            if replicas > 0
        }

    def __str__(self):
        return '%i/%i' % (self.first, self.total)

    def __repr__(self):
        return str(self)


class _ReplicationState2(object):
    def __init__(self, all, first, total):
        self.all = all
        self.first = first
        self.total = total


class _NewDeployerLocation(object):
    def __init__(
            self,
            namespace_filter,
            deploy_url,
    ):
        self.namespace_filter = namespace_filter
        self.deploy_url = deploy_url
        self._cache = {}

    # cache to update all light deployer stats frequently
    @staticmethod
    @memoize.get_cache(records_limit=200)
    def fetch_new_deployer_url(url):
        res = utils.retry_get_json_by_url(url)
        return memoize.SaveWithTtl(res, ttl=2 * 60)

    def update(self):
        try:
            data = self.fetch_new_deployer_url(self.deploy_url)
            self._cache = {
                namespace: state for namespace, state in data.items()
                if self.namespace_filter(namespace)
            }
        except Exception:
            raise PartialUpdate()

    @property
    def deploy_replicas_state(self):
        result = {}
        for namespace, state in self._cache.iteritems():
            generation = utils.get_generation_from_namespace(namespace)
            result[generation] = _ReplicationState2(
                state['replicas']['percentage'] * state['resources']['total'],
                state['resources']['done'],
                state['resources']['total'],
            )
        return result

    @property
    def search_replicas_state(self):
        return collections.defaultdict(_ReplicationState)

    @property
    def location_deploy_url(self):
        return '/'.join(self.deploy_url.split('/')[:5]) + '/'


class _Location(object):
    def __init__(
            self,
            shard_template,
            deploy_url,
            search_url,
    ):
        self.deploy_url = deploy_url
        self.search_url = search_url
        self._shard_template = shard_template
        self._deploy_cache = {}
        self._search_cache = {}

    def update(self):
        try:
            self._update_deploy_cache()
            self._update_search_cache()
        except Exception:
            raise PartialUpdate()

    def _update_deploy_cache(self):
        data = utils.retry_get_json_by_url(self.deploy_url)
        if isinstance(data, dict):
            self._deploy_cache = {
                shard_name: state
                for shard_name, state in data.items()
                if self._shard_template in shard_name
            }
        else:
            self._deploy_cache = {
                item['id']['name']: dict(item, building=item['downloading'])
                for item in data
                if self._shard_template in item['id']['name']
            }

    def _update_search_cache(self):
        cache = {}
        for slot_state in utils.retry_get_json_by_url(self.search_url).values():
            cache.update({
                instance: state
                for instance, state in slot_state.items()
                if self._shard_template in (state['target'] or '')
            })
        self._search_cache = cache

    @property
    def deploy_replicas_state(self):
        result = {}
        for shard_name, state in self._deploy_cache.items():
            generation = utils.get_generation_by_shard(shard_name)
            result.setdefault(generation, _ReplicationState())
            all_ = state.get('building', []) + state.get('idle', []) + state.get('dead', []) + state.get('prepared', [])
            result[generation].add_shard(len(state['prepared']), len(all_))
        return result

    @property
    def search_replicas_state(self):
        shards_state = collections.defaultdict(lambda: dict(total=0, active=0))
        for instance, state in self._search_cache.items():
            if state['target']:
                shards_state[state['target']]['total'] += 1
                shards_state[state['target']]['active'] += 1 if state['target'] == state['observed'] else 0
        result = collections.defaultdict(_ReplicationState)
        for shard_name, state in shards_state.items():
            generation = utils.get_generation_by_shard(shard_name)
            result[generation].add_shard(state['active'], state['total'])
        return result

    @property
    def location_deploy_url(self):
        return '/'.join(self.deploy_url.split('/')[:5]) + '/'


class _Builder(object):
    def __init__(
            self,
            url,
            tier_name,
            fake_empty_builder=False
    ):
        self.url = url
        self.tier_name = tier_name
        self._cache = {}
        self._fake_empty_builder = fake_empty_builder

    def update(self):
        if self._fake_empty_builder:
            return

        try:
            self._cache = utils.retry_get_json_by_url(self.url) or {}
        except Exception:
            raise PartialUpdate()

    @property
    def build_state(self):
        return self._cache

    @property
    def viewer_url(self):
        if 'web/prod' in self.url:
            url = 'http://ctrl.clusterstate.yandex-team.ru/web/prod/jupiter/builder_{}'.format(self.tier_name)
            return url + '?viewer=da&handler=/build_progress'
        return self.url


def _build_url(base_url, tier_name):
    return '{}/view/build_progress/{}'.format(base_url, tier_name)


def _deploy_url(base_url, tier_name):
    return '{}/view/deploy_progress/{}'.format(base_url, tier_name)


def _deploy_url2(base_url):
    return '{}/view/deploy_namespace_percentage'.format(base_url)


def _search_url(base_url, slot_name=None):
    if slot_name:
        return '{}/view/searchers_state/{}'.format(base_url, slot_name)
    return '{}/view/searchers_state'.format(base_url)


def make_jupiter():
    _ctrl_url = 'http://ctrl.clusterstate.yandex-team.ru'
    main_url = _ctrl_url + '/web/prod'
    slot_urls = {
        'pip': _ctrl_url + '/web/pip',
        'sas': _ctrl_url + '/web/prod-sas',
        'vla': _ctrl_url + '/web/prod-vla',
    }

    def _make_tier(tier_name, ctrl_tier_name=None, shard_template=None, slots=None, namespace_filter=None, fake_empty_builder=False):
        ctrl_tier_name = ctrl_tier_name or tier_name
        shard_template = shard_template or ctrl_tier_name
        return _Tier(
            tier_name=tier_name,
            builder=(_Builder(_build_url(main_url, ctrl_tier_name), tier_name, fake_empty_builder=fake_empty_builder)),
            locations=utils.sort_dict({
                slot_name: (
                    _NewDeployerLocation(
                        namespace_filter=(namespace_filter or (lambda namespace: '/' + ctrl_tier_name + '/' in namespace)),
                        deploy_url=_deploy_url2(url if tier_name != 'PlatinumTier' else url.replace('sas', 'sas-platinum')),
                    ) if (slot_name != 'pip' or tier_name in ('WebTier0', 'Tier0 RS', 'WebTier1', 'Tier1 RS')) else _Location(
                        shard_template=shard_template,
                        deploy_url=_deploy_url(url, ctrl_tier_name),
                        search_url=_search_url(url),
                    )
                )
                for slot_name, url in slot_urls.items()
                if slot_name in (slots or slot_urls)
            }, order=['pip', 'sas', 'vla'])
        )

    tiers = (
        _make_tier('MsUserDataTier', 'MsuseardataJupiterTier0', 'rearr'),
        _make_tier('PlatinumTier', 'PlatinumTier0'),
        _make_tier('WebTier0', namespace_filter=(lambda namespace: '/yt/' not in namespace and '/WebTier0/' in namespace)),
        _make_tier('WebTier0Mini', namespace_filter=(lambda namespace: '/yt/' not in namespace and '/WebTier0Mini/' in namespace)),
        _make_tier('AttributeWebTier0', namespace_filter=(lambda namespace: '/AttributeWebTier0/' in namespace)),
        _make_tier('Tier0 RS', namespace_filter=(lambda namespace: '/yt/' in namespace and '/WebTier0/' in namespace), fake_empty_builder=True),
        _make_tier('KeyInvWebTier0', fake_empty_builder=True),
        _make_tier('Gemini', 'GeminiTier', slots={'sas', 'vla'}),
        _make_tier('Castor', 'CastorTier', slots={'sas', 'vla'}),
        _make_tier('JudTier', slots={'vla',}),
        _make_tier('EmbeddingWebTier0', fake_empty_builder=True),
        _make_tier('InvertedIndexWebTier0', fake_empty_builder=True),
    )
    tiers_order = [tier.tier_name for tier in tiers]
    tiers = utils.sort_dict({tier.tier_name: tier for tier in tiers}, tiers_order)
    ctrls_checker = _ControllersStateChecker(dict(slot_urls, main=main_url))
    return _Source(
        tiers=tiers,
        ctrls_checker=ctrls_checker,
    )


def make_video():
    ctrl_url = 'http://ctrl.clusterstate.yandex-team.ru/video'
    main_url = ctrl_url + '/prod/'
    slot_urls = {
        'pip': ctrl_url + '/pip',
        'man': ctrl_url + '/prod-man',
        'sas': ctrl_url + '/prod-sas',
        'vla': ctrl_url + '/prod-vla',
    }

    def _make_tier(tier_name, ctrl_tier_name=None, shard_template=None, slots=None):
        ctrl_tier_name = ctrl_tier_name or tier_name
        shard_template = shard_template or ctrl_tier_name
        return _Tier(
            tier_name=tier_name,
            builder=_Builder(_build_url(main_url, ctrl_tier_name), ctrl_tier_name),
            locations=utils.sort_dict({
                slot_name: _Location(
                    shard_template=shard_template,
                    deploy_url=_deploy_url(url, ctrl_tier_name),
                    search_url=_search_url(url),
                )
                for slot_name, url in slot_urls.items()
                if slot_name in (slots or slot_urls)
            }, order=['pip', 'man', 'sas', 'vla'])
        )

    tiers = (
        _make_tier('VideoPlatinum', shard_template='vidsidxpt-'),
        _make_tier('VideoTier0', shard_template='vidsidx-'),
        _make_tier('VideoEmbeddingPlatinum', shard_template='VideoEmbeddingPlatinum-'),
        _make_tier('VideoEmbeddingTier0', shard_template='VideoEmbeddingTier0-'),
        _make_tier('VideoInvertedPlatinum', shard_template='VideoInvertedPlatinum-'),
        _make_tier('VideoInvertedTier0', shard_template='VideoInvertedTier0-'),
    )
    tiers = utils.sort_dict(
        {tier.tier_name: tier for tier in tiers},
        [tier.tier_name for tier in tiers])
    return _Source(
        tiers=tiers,
        ctrls_checker=_ControllersStateChecker(dict(slot_urls, main=main_url)),
    )


def make_rim():
    ctrl_url = 'http://ctrl.clusterstate.yandex-team.ru/rim'
    main_url = ctrl_url + '/prod/'
    slot_urls = {
        'pip': ctrl_url + '/pip',
        # 'man': ctrl_url + '/man',
        'sas': ctrl_url + '/sas',
        'vla': ctrl_url + '/vla',
    }

    def _make_tier(tier_name, ctrl_tier_name=None, shard_template=None, slots=None):
        ctrl_tier_name = ctrl_tier_name or tier_name
        shard_template = shard_template or ctrl_tier_name
        return _Tier(
            tier_name=tier_name,
            builder=_Builder(_build_url(main_url, ctrl_tier_name), ctrl_tier_name),
            locations=utils.sort_dict({
                slot_name: _Location(
                    shard_template=shard_template,
                    deploy_url=_deploy_url(url, ctrl_tier_name),
                    search_url=_search_url(url),
                )
                for slot_name, url in slot_urls.items()
                if slot_name in (slots or slot_urls)
            }, order=['pip', 'sas', 'vla'])
        )

    tiers = (
        _make_tier('ImgRIMTier', ctrl_tier_name='ImgsRim3k', shard_template='imgsrim3k-'),
    )

    tiers = utils.sort_dict(
        {tier.tier_name: tier for tier in tiers},
        [tier.tier_name for tier in tiers])

    return _Source(
        tiers=tiers,
        ctrls_checker=_ControllersStateChecker(dict(slot_urls, main=main_url)),
    )


def make_images_commercial():
    ctrl_url = 'http://ctrl.clusterstate.yandex-team.ru/imgcommercial'
    main_url = ctrl_url + '/prod'
    slot_urls = {
        'pip': ctrl_url + '/pip',
        # 'man': ctrl_url + '/man',
        'sas': ctrl_url + '/sas',
        'vla': ctrl_url + '/vla',
    }

    def _make_tier(tier_name, ctrl_tier_name=None, shard_template=None, slots=None):
        ctrl_tier_name = ctrl_tier_name or tier_name
        shard_template = shard_template or ctrl_tier_name
        return _Tier(
            tier_name=tier_name,
            builder=_Builder(_build_url(main_url, ctrl_tier_name), ctrl_tier_name),
            locations=utils.sort_dict({
                slot_name: _Location(
                    shard_template=shard_template,
                    deploy_url=_deploy_url(url, ctrl_tier_name),
                    search_url=_search_url(url),
                )
                for slot_name, url in slot_urls.items()
                if slot_name in (slots or slot_urls)
            }, order=['pip', 'sas', 'vla'])
        )

    # values from infra.callisto.controllers.sdk.tier
    sdk_tier = collections.namedtuple('SdkTier', ['name', 'prefix'])(name='ImgCommercialTier0', prefix='img_commercial')

    tiers_list = (
        _make_tier(sdk_tier.name, ctrl_tier_name='ImgCommercialTier0', shard_template=sdk_tier.prefix + '-'),
    )

    tiers = utils.sort_dict(
        {tier.tier_name: tier for tier in tiers_list},
        [tier.tier_name for tier in tiers_list])

    return _Source(
        tiers=tiers,
        ctrls_checker=_ControllersStateChecker(dict(slot_urls, main=main_url)),
    )


def make_images():
    ctrl_url = 'http://ctrl.clusterstate.yandex-team.ru/img'
    main_url = ctrl_url + '/prod/'
    slot_urls = {
        'pip': ctrl_url + '/pip',
        # 'man': ctrl_url + '/prod-man',
        'sas': ctrl_url + '/prod-sas',
        'vla': ctrl_url + '/prod-vla',
    }

    def _make_tier(tier_name, ctrl_tier_name=None, shard_template=None, slots=None, namespace_filter=None, fake_empty_builder=False):
        ctrl_tier_name = ctrl_tier_name or tier_name
        shard_template = shard_template or ctrl_tier_name
        return _Tier(
            tier_name=tier_name,
            builder=(_Builder(_build_url(main_url, ctrl_tier_name), tier_name, fake_empty_builder=fake_empty_builder)),
            locations=utils.sort_dict({
                slot_name: (
                    _NewDeployerLocation(
                        namespace_filter=(namespace_filter or (lambda namespace: '/' + ctrl_tier_name + '/' in namespace)),
                        deploy_url=_deploy_url2(url),
                    ) if (slot_name != 'pip' or tier_name in ('ImgTier0 RS', 'ImgTier0')) else _Location(
                        shard_template=shard_template,
                        deploy_url=_deploy_url(url, ctrl_tier_name),
                        search_url=_search_url(url),
                    )
                )
                for slot_name, url in slot_urls.items()
                if slot_name in (slots or slot_urls)
            }, order=['pip', 'sas', 'vla'])
        )

    tiers = (
        _make_tier('ImgTier0', shard_template='imgsidx-', namespace_filter=(lambda namespace: '/yt/' not in namespace and '/ImgTier0/' in namespace)),
        _make_tier('ImgTier1', shard_template='imgsidxt1-'),
        _make_tier('ImgEmbeddingTier0', ctrl_tier_name='EmbeddingImgTier0', shard_template='imgsembedding-'),
        _make_tier('ImgInvIndexTier0', ctrl_tier_name='InvertedIndexImgTier0', shard_template='imgsinverted-'),
        _make_tier('ImgMmetaTier0', shard_template='imgmmeta-'),
        _make_tier('ImgTier0 RS', slots={'pip', 'sas'}, namespace_filter=(lambda namespace: '/yt/' in namespace and '/ImgTier0/' in namespace), fake_empty_builder=True),
    )

    tiers = utils.sort_dict(
        {tier.tier_name: tier for tier in tiers},
        [tier.tier_name for tier in tiers])

    return _Source(
        tiers=tiers,
        ctrls_checker=_ControllersStateChecker(dict(slot_urls, main=main_url)),
    )


def make_plain(url, tier_name, shard_template=None):
    shard_template = shard_template or tier_name
    tiers = {
        tier_name: _Tier(
            tier_name,
            _Builder(_build_url(url, tier_name)),
            {
                'slot': _Location(
                    shard_template,
                    _deploy_url(url, tier_name),
                    _search_url(url),
                )
            }
        )
    }
    ctrl_checkers = _ControllersStateChecker({'main': url})
    return _Source(
        tiers=tiers,
        ctrls_checker=ctrl_checkers,
    )


def make_builder_only(url, tier_name, shard_template=None):
    tiers = {
        tier_name: _Tier(
            tier_name,
            _Builder(_build_url(url, tier_name)),
            {}
        )
    }
    ctrl_checkers = _ControllersStateChecker({'main': url})
    return _Source(
        tiers=tiers,
        ctrls_checker=ctrl_checkers,
    )


def make_multi(url, tier_names, shard_templates=None):
    def _make_tier(tier_name, shard_template=None):
        ctrl_tier_name = tier_name
        shard_template = shard_template or ctrl_tier_name
        return _Tier(
            tier_name=tier_name,
            builder=_Builder(_build_url(url, ctrl_tier_name)),
            locations={
                'slot': _Location(
                    shard_template=shard_template,
                    deploy_url=_deploy_url(url, ctrl_tier_name),
                    search_url=_search_url(url),
                )
            }
        )
    tiers = {
        tier_name: _make_tier(tier_name, shard_templates.get(tier_name))
        for tier_name in tier_names
    }
    ctrl_checkers = _ControllersStateChecker({'main': url})
    return _Source(
        tiers=tiers,
        ctrls_checker=ctrl_checkers,
    )


def make_callisto():
    sas_url = 'http://ctrl.clusterstate.yandex-team.ru/callisto/sas'
    vla_url = 'http://ctrl.clusterstate.yandex-team.ru/callisto/vla'
    tier_name = 'WebFreshTier'
    locations = {
        'vla': _Location(
            tier_name,
            _deploy_url(vla_url, tier_name),
            _search_url(vla_url),
        ),
        'sas': _Location(
            tier_name,
            _deploy_url(sas_url, tier_name),
            _search_url(sas_url),
        )
    }
    tier = _Tier(
        tier_name=tier_name,
        builder=_Builder(_build_url(vla_url, tier_name), tier_name),
        locations=utils.sort_dict(locations, order=['sas', 'vla'])
    )

    ctrls_checker = _ControllersStateChecker({'sas': sas_url, 'vla': vla_url})
    return _CallistoSource(
        tiers={tier.tier_name: tier},
        ctrls_checker=ctrls_checker,
    )


def make_img_thumb():
    url = 'https://ctrl.clusterstate.yandex-team.ru/img/thumb'
    return _Source({}, _ControllersStateChecker({'all': url}))
