import collections
import datetime
import logging
import math
import os
import time

import gevent.event
import retry

import infra.callisto.protos.deploy.tables_pb2 as tables  # noqa

import chaos
import notification
import resource as deploy_resource
import resource_pool
import storage
import unistat
import yt.wrapper as yt


class Deployer(object):
    DEFAULT_COOLDOWN = 120
    TOLERABLE_EXCEPTIONS = (IOError, OSError, RuntimeError, yt.YtResponseError)
    _resource_pool = None  # type: resource_pool.ResourcePool

    def __init__(self, pod_id, storage_root, target_client, status_client, config, notifications_config, use_new_rescan):
        self._pod_id = pod_id
        self._target_client = target_client
        self._status_client = status_client
        self._config = config
        self._namespace_notifications = {x.Namespace: x.Notification for x in notifications_config.Notifications}
        self._use_new_rescan = use_new_rescan
        self._ns = {x.Namespace: x for x in notifications_config.Notifications}
        self._ns_notified = {x.Namespace: datetime.datetime.min for x in notifications_config.Notifications}

        self._storage = storage.Storage(storage_root, cleanup_cooldown=config.Execution.CleanupCooldown / 1000.0, deferred_remove_iterations=config.Execution.DeferredRemoveIterations)
        self._download_greenlet = None

        self._last_reload_plutonium_config = None
        self._reload_plutonium_config()

    def start(self):
        log = _log.getChild('startup')

        self._resource_pool = self._read_resources()
        log.info('%s prepared resources found', len(self._resource_pool))

        log.info('Trying to receive targets')
        delay = 1.0
        backoff = 1.5
        deadline = self._config.Startup.TargetReadDeadline / 1000.0
        tries = min(_exponential_tries(delay, backoff, deadline), 1)
        try:
            targets = retry.retry_call(self._read_targets,
                                       exceptions=Deployer.TOLERABLE_EXCEPTIONS,
                                       delay=delay, backoff=backoff, tries=tries,
                                       logger=log)
        except Deployer.TOLERABLE_EXCEPTIONS:
            log.exception('Unable to receive targets with {} attempts'.format(tries))
        else:
            log.info('%s targets read', len(targets))
            self._resource_pool.reset(targets)

        self._set_readiness()

        ready_resources = self._resource_pool.get_ready()
        if ready_resources:
            log.info('Trying to notify with prepared resources')
            delay = 1.0
            backoff = 1.5
            deadline = self._config.Startup.NotificationDeadline / 1000.0
            tries = min(_exponential_tries(delay, backoff, deadline), 1)
            resource, target = ready_resources.iteritems().next()
            try:
                retry.retry_call(self._notify, fargs=[resource, target, ready_resources],
                                 exceptions=Deployer.TOLERABLE_EXCEPTIONS,
                                 delay=delay, backoff=backoff, tries=tries,
                                 logger=log)
            except Deployer.TOLERABLE_EXCEPTIONS:
                log.exception('Unable to notify with {} attempts'.format(tries))
            self._notify(resource, target, ready_resources)
        else:
            log.info('Storage is empty')

        log.info('Spawn regular routines')
        self._restart_download_loop()
        gevent.spawn(self._main_loop)

        log.info('Done')

    def _set_readiness(self):
        ready_count = len(self._resource_pool.get_ready())
        total_count = len(self._resource_pool)
        ready = ready_count >= total_count * self._config.Startup.ReadinessRatio
        _log.info('ready %s of %s while required %s',
                  ready_count, total_count, total_count * self._config.Startup.ReadinessRatio)
        unistat.set_ready(ready)

    def _set_targets(self, targets):
        if frozenset(self._resource_pool.keys()) != frozenset(targets.iterkeys()):
            _log.info('Targets differ')
            self._resource_pool.reset(targets)
        else:
            _log.debug('Targets do not differ')

        unistat.set_counter('ready_resources_count_ammm', self._resource_pool.ready_count())
        unistat.set_counter('not_ready_resources_count_ammm', self._resource_pool.not_ready_count())
        unistat.set_counter('not_ready_resources_count_axxx', self._resource_pool.not_ready_count())

        if self._storage.current_resource is not None and self._storage.current_resource not in self._resource_pool:
            _log.debug('Current resource %s was deleted from targets',
                       self._storage.current_resource)
            self._restart_download_loop()

    def _read_targets(self):
        chaos.produce()

        targets_by_resource = {}
        for target in self._target_client.load_targets(self._pod_id):
            resource = deploy_resource.ResolvedResource(namespace=target.Namespace,
                                                        name=target.LocalPath,
                                                        rbtorrent=None, size=0)
            targets_by_resource[resource] = target
        return targets_by_resource

    def _read_resources(self):
        pool = resource_pool.ResourcePool()
        for resource_key in self._storage.list_ready():
            _log.debug('Resource %s is ready', resource_key)
            description = self._storage.load_description(resource_key)
            pool.add_existing(resource_key, description.Target)
        return pool

    def _download_loop(self):
        _log.info('Start iterations')
        while True:
            try:
                self._cleanup()
                self._notify_namespaces()
                self._reload_plutonium_config()

                for resource, target in self._resource_pool.list():
                    try:
                        if resource not in self._resource_pool:
                            continue

                        if not self._storage.is_resource_ready(resource):
                            self._storage.invalidate(resource)
                            self._reload_plutonium_config()
                            self._download(resource, target)
                            self._set_readiness()
                        self._notify(resource, target)
                    except Deployer.TOLERABLE_EXCEPTIONS:
                        _log.exception('Tolerable exception in download_loop')
                        self._notify_namespaces()
                    except Exception:  # noqa
                        _log.exception('Unhandled exception in download_loop')
                        os._exit(1)  # noqa
            finally:
                gevent.sleep(self._config.Execution.DownloadInterval / 1000.0)

    def _main_loop(self):
        while True:
            try:
                targets = self._read_targets()
                _log.debug('Loaded %s targets', len(targets))
                unistat.set_counter('targets_count_ammm', len(targets))
                self._set_targets(targets)

                statuses = self._collect_statuses()
                _log.debug('Collected %s statuses', len(statuses))
                self._update_status_counters(statuses)
                self._status_client.write_statuses(self._pod_id, statuses)
            except Deployer.TOLERABLE_EXCEPTIONS:
                _log.exception('Tolerable exception in main_loop')
            except Exception:  # noqa
                _log.exception('Unhandled exception in main_loop')
                os._exit(1)  # noqa
            finally:
                gevent.sleep(self._config.Execution.UpdateTargetInterval / 1000.0)

    def _collect_statuses(self):
        return [tables.TPodStatus(PodId=self._pod_id,
                                  Namespace=resource.namespace,
                                  LocalPath=resource.name,
                                  ResourceState=tables.TResourceState(Status=status))
                for resource, status in self._resource_pool.get_statuses().iteritems()]

    def _update_status_counters(self, statuses):
        unistat.set_counter('statuses_count_ammm', len(statuses))

        counter = collections.Counter(status.ResourceState.Status for status in statuses)
        for key, descriptor in tables.EDownloadState.DESCRIPTOR.values_by_number.iteritems():
            sig = 'resources_{}_ammm'.format(descriptor.name)
            unistat.set_counter(sig, counter[key])
        unistat.set_counter('storage_total_bytes_ammv', self._storage.total_space())
        unistat.set_counter('storage_free_bytes_ammv', self._storage.free_space())

    def _download(self, resource, target):
        start_downloading_resource = time.time()
        unistat.inc_counter('download_resource_count_dmmm')

        _log.debug('Try to download\n%s', resource)
        self._resource_pool.set_status(resource, tables.EDownloadState.DOWNLOADING)
        try:
            chaos.produce()
            self._storage.download(target, resource)
            self._resource_pool.set_status(resource, tables.EDownloadState.PREPARED)
        except Exception:
            self._resource_pool.set_status(resource, tables.EDownloadState.IDLE)
            unistat.inc_counter('download_resource_fail_dmmm')
            raise

        _log.info('Downloaded %s', resource)
        unistat.inc_counter('download_resource_success_dmmm')
        unistat.push_time_s_histogram('download_resource_time_dhhh', time.time() - start_downloading_resource)

    def _notify(self, resource, target, ready_resources=None):
        try:
            spec = target.Notification
            if spec.WhichOneof('Action') is None:
                spec = self._namespace_notifications.get(resource.namespace, spec)
            if spec.WhichOneof('Action') is None:
                return

            chaos.produce()

            if not ready_resources:
                ready_resources = self._resource_pool.get_ready()

            if self._cooldown(resource.namespace):
                _log.debug('Try to notify %s', resource)
                if self._use_new_rescan:
                    notification_result = notification.rescan_resources(spec, ready_resources, storage=self._storage)
                else:
                    notification_result = notification.notify_resource(resource, spec, ready_resources.keys())

                _log.info('Notified %s resources', len(notification_result))
                for resource, status in notification_result.iteritems():
                    enum_state = tables.EDownloadState.ACTIVE if status else tables.EDownloadState.PREPARED
                    self._resource_pool.set_status(resource, enum_state)
        except IOError:
            pass

    def _notify_namespace(self, namespace, ready_resources):
        spec = self._namespace_notifications.get(namespace)
        if not spec:
            return

        if self._cooldown(namespace):
            _log.debug('Try to notify namespace %s', namespace)
            notification_result = notification.rescan_resources(spec, ready_resources, storage=self._storage)
            _log.info('Notified %s resources', len(notification_result))

            for resource, status in notification_result.iteritems():
                enum_state = tables.EDownloadState.ACTIVE if status else tables.EDownloadState.PREPARED
                self._resource_pool.set_status(resource, enum_state)

    def _cooldown(self, namespace):
        now = datetime.datetime.now()
        self._ns_notified.setdefault(namespace, datetime.datetime.min)
        if (now - self._ns_notified[namespace]).total_seconds() < self._get_cooldown(namespace):
            return False

        self._ns_notified[namespace] = now
        return True

    def _get_cooldown(self, namespace):
        n = self._ns.get(namespace)
        cooldown = n.Cooldown if n else Deployer.DEFAULT_COOLDOWN
        return cooldown or Deployer.DEFAULT_COOLDOWN

    def _restart_download_loop(self):
        _log.info('Restarting download loop')
        if self._download_greenlet is not None:
            self._download_greenlet.kill(timeout=30)
            if not self._download_greenlet.dead:
                _log.fatal('could not stop download loop, exiting')
                os._exit(1)  # noqa
        self._download_greenlet = gevent.spawn(self._download_loop)

    def _notify_namespaces(self):
        try:
            if self._use_new_rescan and self._ns:
                ready_resources = self._resource_pool.get_ready()
                self._notify_namespace(self._ns.iterkeys().next(), ready_resources)
        except Deployer.TOLERABLE_EXCEPTIONS as e:
            _log.exception(e)

    def _cleanup(self):
        try:
            _log.debug('cleanup')
            self._storage.cleanup_except(set(self._resource_pool.keys()))

        except Deployer.TOLERABLE_EXCEPTIONS as e:
            _log.exception(e)

    def _reload_plutonium_config(self):
        if self._config.PlutoniumConfigCluster and self._config.PlutoniumConfigPath:
            import infra.callisto.deploy.deployer.plutonium as plutonium  # noqa
            import yt.wrapper as yt

            ts = time.time()
            if (self._last_reload_plutonium_config is None) or (ts - self._last_reload_plutonium_config >= 10):
                self._last_reload_plutonium_config = ts
                _log.info('Reloading Plutonium config...')
                plutonium.reload_global_config(self._config.PlutoniumConfigCluster, yt._get_token(), self._config.PlutoniumConfigPath)


def _exponential_tries(delay, backoff, deadline):
    return int(math.log(deadline * (backoff - 1) / delay + 1, backoff))


_log = logging.getLogger(__name__)
