import gevent
import logging
import os
import time

import infra.callisto.deploy.storage.path as path
import infra.callisto.libraries.copier as copier

import unistat


def download_resource(storage, namespace, resource_spec, destination, existing=None):
    source_type = resource_spec.WhichOneof('Source')
    if source_type == 'Static':
        _static(resource_spec.Static, destination)
    elif source_type == 'DynamicTables':
        _plutonium(resource_spec.DynamicTables, destination, existing)
    elif source_type == 'Compound':
        _download_compound(storage, namespace, resource_spec.Compound, destination, existing)
    elif source_type == 'Skynet':
        _skynet(storage, resource_spec.Skynet, destination)
    else:
        # todo: should result in appropriate status record instead of the whole deployer failing
        raise RuntimeError('Unsupported source type: {}'.format(source_type))


def _skynet(storage, spec, destination):
    _log.info('Download skynet %s -> %s', spec.Url, destination)
    copier.sky_get(
        resource_url=spec.Url,
        dst_dir=destination,
        container_params=_skyget_container_params(spec, storage.root()),
        copier_opts='{direct_write: 1, subproc: 1, max_write_chunk: 131072}',
        hardlink=True
    )


def _skyget_container_params(spec, storage_root):
    stat = os.stat(storage_root)
    io_limit = spec.IoLimit or '49M'
    return {
        'io_limit': '{}\\:{}: {}'.format(os.major(stat.st_dev), os.minor(stat.st_dev), io_limit),
        'io_policy': 'idle',
        'io_weight': 0.01
    }


def _static(spec, destination):
    _log.info('Write static resource')
    with open(destination, 'w') as f:
        f.write(spec.Content)
        os.fsync(f.fileno())


def _plutonium(spec, destination, existing):
    if existing and spec.RowId in existing:
        existing_resource = existing[spec.RowId]
        _log.debug('Link existing plutonium %s#%s: %s -> %s', spec.Path, spec.RowId, existing_resource, destination)
        try:
            os.unlink(destination)
            _log.debug('Destination %s already exists, removing', destination)
        except OSError:
            pass
        os.link(existing_resource, destination)
    else:
        locations = [spec]
        for location in spec.FallbackLocations:
            locations.append(location)
        _download_plutonium(locations, spec, destination)
        if existing:
            existing[spec.RowId] = destination

    # todo: should fsync containing directory as well.
    with open(destination) as f:
        os.fsync(f.fileno())


def _download_plutonium(locations, spec, destination):
    row_id = spec.RowId
    row_weight = spec.RowWeight if spec.RowWeight else 400  # TODO: replace with 1 after the shard-ctrl is released

    # Imports are hidden for builds without links to plutonium (and YT by further references).
    import infra.callisto.deploy.deployer.plutonium as plutonium  # noqa

    _log.debug('Download plutonium %s -> %s', row_id, destination)

    for location in locations:
        start_download_time = time.time()
        try:
            _log.debug('Try %s / %s', location.Cluster, location.Path)
            gevent.get_hub().threadpool.apply(plutonium.download, args=(location.Cluster,
                                                                        location.Path,
                                                                        row_id,
                                                                        destination,
                                                                        row_weight))
            success_download_time = (time.time() - start_download_time) * 1000
            _log.debug('Finished downloading plutonium %s -> %s in %s ms', row_id, destination, success_download_time)
            unistat.push_time_ms_histogram('download_plutonium_time_ms_dhhh', success_download_time)
            unistat.push_time_ms_histogram('download_plutonium_{}_time_ms_dhhh'.format(location.Cluster.replace('-', '_')), success_download_time)
            return
        except RuntimeError:
            fail_download_time = (time.time() - start_download_time) * 1000
            _log.exception('Failed to download plutonium %s -> %s, time spent: %s ms', row_id, destination, fail_download_time)
            unistat.push_time_ms_histogram('fail_download_plutonium_time_ms_dhhh', fail_download_time)
            unistat.push_time_ms_histogram('fail_download_plutonium_{}_time_ms_dhhh'.format(location.Cluster.replace('-', '_')), fail_download_time)
            unistat.inc_counter('plutonium_read_failures_dmmm')
    raise RuntimeError('Unable to download {}'.format(row_id))


def _download_compound(storage, namespace, spec, destination, existing):
    _log.info('Begin download compound -> %s', destination)
    if not existing:
        existing = _load_existing(storage, namespace)
    path.ensure_dir(destination)
    for internal_resource in spec.Sources:
        internal_destination = os.path.join(destination, internal_resource.Path)
        path.ensure_dir(os.path.dirname(internal_destination))
        download_resource(storage, namespace, internal_resource.Source, internal_destination, existing)
    _log.info('Successfully downloaded compound -> %s', destination)


# Compound of compounds are not supported deliberately.
def _load_existing(storage, namespace):
    existing = {}
    for resource_key in storage.list_ready():
        target = storage.load_description(resource_key).Target
        if target.Namespace != namespace:
            continue
        if target.ResourceSpec.WhichOneof('Source') == 'Compound':
            for source in target.ResourceSpec.Compound.Sources:
                if source.Source.WhichOneof('Source') != 'DynamicTables':
                    continue
                existing[source.Source.DynamicTables.RowId] = os.path.join(storage.realpath(resource_key), source.Path)
        elif target.ResourceSpec.WhichOneof('Source') == 'DynamicTables':
            existing[target.ResourceSpec.DynamicTables.RowId] = storage.realpath(resource_key)
    return existing


_log = logging.getLogger(__name__)
