import logging

import yt.wrapper as yt
import infra.callisto.protos.deploy.tables_pb2 as tables  # noqa
import search.plutonium.deploy.proto.rescan_pb2 as rescan  # noqa
import search.plutonium.deploy.proto.sources_pb2 as sources  # noqa

import utils


def _key(status):
    return (status.PodId, status.Namespace, status.LocalPath)


class _Client(object):
    def __init__(self, yt_client, table, notify_table):
        self.yt_client = yt_client
        self.table = table
        self.notify_table = notify_table

    def load_targets(self, pod_id):
        try:
            return list(_select_pod_targets(self.yt_client, self.table, pod_id))
        except yt.errors.YtResponseError:
            _log.exception('Unable to receive response from yt')
            raise RuntimeError('Unable to load targets')

    def write_statuses(self, pod_id, statuses):
        with yt.Transaction(client=self.yt_client, type='tablet'):
            current_rows = {(status['PodId'], status['Namespace'], status['LocalPath']): status['ResourceState'] for status in _select_status(self.yt_client, self.table, pod_id)}
            new_rows = {_key(status): status.ResourceState.SerializeToString() for status in statuses}

            diff = utils.diff(current_rows, new_rows)

            if diff.insert or diff.remove:
                _log.debug('Statuses differ')

                if diff.remove:
                    self._remove_keys(diff.remove)
                if diff.insert:
                    self._insert_dict(diff.insert)
                self._append_notification(pod_id)
            else:
                _log.debug('Statuses do not differ')

    def _remove_keys(self, keys):
        _log.debug('remove %s statuses', len(keys))
        rows = [{'PodId': key[0], 'Namespace': key[1], 'LocalPath': key[2]} for key in keys]
        self.yt_client.delete_rows(self.table, rows, format='json')

    def _insert_dict(self, dict_):
        _log.debug('insert %s statuses', len(dict_))
        rows = [{'PodId': key[0], 'Namespace': key[1], 'LocalPath': key[2], 'ResourceState': value}
                for key, value in dict_.iteritems()]
        self.yt_client.insert_rows(self.table, rows, update=True)

    def _append_notification(self, pod_id):
        if self.notify_table:
            self.yt_client.insert_rows(table=self.notify_table, input_stream=[{'PodId': pod_id}], raw=False)


def _select_pod_targets(clt, table, pod_id):
    for row in clt.select_rows(
        # 'PodId, Namespace, LocalPath, ResourceSpec, Notification from [{table}] where PodId="{pod_id}"'.format(
        '* from [{table}] where PodId="{pod_id}"'.format(
            table=table,
            pod_id=pod_id)):
        notification = row.get('Notification')
        if notification is not None:
            notification = tables.TNotification.FromString(notification)

        resource_labels = row.get('ResourceLabels')
        if resource_labels is not None:
            resource_labels = rescan.TResourceLabels.FromString(resource_labels)

        download_policy = row.get('DownloadPolicy')
        if download_policy:
            download_policy = tables.TDownloadPolicy.FromString(download_policy)
        else:
            download_policy = tables.TDownloadPolicy()

        yield tables.TPodTarget(PodId=pod_id,
                                Namespace=row['Namespace'],
                                LocalPath=row['LocalPath'],
                                ResourceSpec=sources.TSource.FromString(row['ResourceSpec']),
                                ResourceLabels=resource_labels,
                                Notification=notification,
                                DownloadPolicy=download_policy)


def _select_status(client, table, pod_id):
    request = 'PodId, Namespace, LocalPath, ResourceState from [{table}] where PodId="{pod_id}"'
    return client.select_rows(request.format(table=table, pod_id=pod_id))


def make_client(proxy, table):
    yt_client = yt.client.YtClient(proxy, config=yt.config.config)
    yt_client.config['backend'] = 'rpc'  # noqa

    return _Client(yt_client, table, notify_table=None)


def make_status_client(proxy, status_table, notify_table):
    yt_client = yt.client.YtClient(proxy, config=yt.config.config)
    yt_client.config['backend'] = 'rpc'  # noqa

    return _Client(yt_client, status_table, notify_table)


_log = logging.getLogger(__name__)
