import gevent.monkey
gevent.monkey.patch_all()  # noqa

import argparse
import json
import logging
import os
import tempfile
import time

import google.protobuf.json_format as json_format
import google.protobuf.text_format as text_format

import infra.callisto.libraries.logging as logging_utils
import infra.callisto.protos.deploy.tables_pb2 as tables  # noqa
import infra.callisto.deploy.deployer.config.config_pb2 as config_pb2  # noqa

from library.python.svn_version import svn_version

from deployer import Deployer
import dynamic_table_client
import unistat


def patch_args(args, config):
    assert not args.debug or not args.no_debug, "cannot set both --debug and --no-debug options"
    assert not args.use_new_rescan or not args.use_old_rescan, "cannot set both --use-new-resacn and --use-old-rescan options"

    args.debug = args.debug or config.Args.Debug and not args.no_debug
    args.use_new_rescan = args.use_new_rescan or config.Args.UseNewRescan and not args.use_old_rescan
    args.log_dir = args.log_dir or config.Args.LogDir or './logs'
    args.pod_id = args.pod_id or config.Args.PodId
    args.storage_root = args.storage_root or config.Args.StorageRoot
    args.yt_proxy = args.yt_proxy or config.Args.YtProxy
    args.target_table = args.target_table or config.Args.TargetTable
    args.status_table = args.status_table or config.Args.StatusTable
    args.notify_table = args.notify_table or config.Args.NotifyTable
    args.target_file = args.target_file or config.Args.TargetFile
    args.status_file = args.status_file or config.Args.StatusFile
    args.unistat_port = args.unistat_port or config.Args.UnistatPort or 8000
    args.notifications_config = args.notifications_config or config.Args.NotificationsConfig

    assert args.pod_id, "--pod-id cannot be empty"
    assert args.storage_root, "--storage-root cannot be empty"


def main():
    args = parse_args()

    if args.svnrevision:
        print(svn_version())
        return

    config = config_pb2.TConfig()
    if args.config:
        with open(args.config) as f:
            config = text_format.Parse(f.read(), config)

    patch_args(args, config)

    logging_utils.configure_logging(args.log_dir, debug=args.debug)

    if config.YtDriverLog:
        configure_yt_driver_logging(logs_dir=args.log_dir, log_name=config.YtDriverLog, debug=args.debug)

    notifications_config = tables.TStaticNotificationConfig()
    if args.notifications_config:
        with open(args.notifications_config) as f:
            notifications_config = text_format.Parse(f.read(), notifications_config)

    deployer = Deployer(args.pod_id, args.storage_root,
                        make_target_client(args),
                        make_status_client(args),
                        config, notifications_config, args.use_new_rescan)
    deployer.start()
    unistat.set_counter('start_time_annn', time.time())
    unistat.serve_forever(args.unistat_port)


def make_target_client(args):
    if args.target_table:
        return dynamic_table_client.make_client(args.yt_proxy, args.target_table)
    elif args.target_file:
        return StaticTargetClient(args.target_file)


def make_status_client(args):
    if args.status_table:
        return dynamic_table_client.make_status_client(args.yt_proxy, args.status_table, args.notify_table)
    elif args.status_file:
        return LocalStatusClient(args.status_file)


class StaticTargetClient(object):
    def __init__(self, filename):
        self.filename = filename

    def load_targets(self, pod_id):
        targets = []
        with open(self.filename) as f:
            for target in json.load(f):
                targets.append(json_format.ParseDict(target, tables.TPodTarget()))
                assert targets[-1].PodId == pod_id
        return targets


class LocalStatusClient(object):
    def __init__(self, filename):
        self.filename = filename

    def write_statuses(self, _, statuses):
        with tempfile.NamedTemporaryFile() as f:
            json.dump([json_format.MessageToDict(status) for status in statuses], f, indent=4)
            f.flush()
            os.fsync(f.fileno())
            try:
                os.unlink(self.filename)
            except OSError:
                pass
            os.link(f.name, self.filename)
            with open(self.filename) as g:
                os.fsync(g.fileno())

        logging.getLogger(self.__class__.__name__).info('Write %s statuses into %s', len(statuses), self.filename)


def parse_args():
    parser = argparse.ArgumentParser(description='Dyntables-driven deployer')
    parser.add_argument('--debug', dest='debug', default=False, action='store_true',
                        help='enable fancy logging to stdout')
    parser.add_argument('--use-new-rescan', default=False, action='store_true',
                        help='enable new protobufs in rescan notify (DynamicStorage)')
    parser.add_argument('--no-debug', dest='no_debug', default=False, action='store_true',
                        help='disable debug mode (if enabled in the config)')
    parser.add_argument('--use-old-rescan', default=False, action='store_true',
                        help='disable new rescan mode (if enabled in the config)')
    parser.add_argument('--log-dir', help='Log dir')
    parser.add_argument('--pod-id')
    parser.add_argument('--storage-root')
    parser.add_argument('--yt-proxy')
    parser.add_argument('--target-table')
    parser.add_argument('--status-table')
    parser.add_argument('--notify-table')
    parser.add_argument('--target-file', help='Local file with list of targets. For testing only!')
    parser.add_argument('--status-file', help='Local file to save status to. For testing only!')
    parser.add_argument('--unistat-port')
    parser.add_argument('--config')
    parser.add_argument('--notifications-config')
    parser.add_argument('--svnrevision', action='store_true', help='print Arcadia version info and exit')
    return parser.parse_args()


def configure_yt_driver_logging(logs_dir, log_name, debug):
    import yt.wrapper as yt
    import os.path as path

    yt.config.config['driver_logging_config'] = {
        "rules": [
            {
                "min_level": "debug" if debug else "info",
                "max_level": "maximum",
                "writers": [
                    "main",
                ],
            },
        ],
        "writers": {
            "main": {
                "type": "file",
                "file_name": path.join(logs_dir, log_name),
                "enable_source_location": True
            },
        },
        "watch_period": 1,
    }
